#! /usr/bin/env python
# mmu2html.py
# Copyright (C) 2013-2014 Ralf Hoffmann.
# ralf@boomerangsworld.de
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import re
import os
import sys
import shutil
import subprocess
from cStringIO import StringIO
import ConfigParser
from optparse import OptionParser

MMU2HTML_VERSION = 0.2

cms_dir = "cms"
files_dir = "files"
output_dir = "html"
base_css = "style.css"
tpl_file = ""

def insert_menu_node( menu, entry, name):
    base = entry[0]
    rest = entry[1:]

    sub_node = None

    for children in menu[2]:
        if children[0] == base:
            sub_node = children
            break

    if sub_node is None:
        if len( rest ) > 0:
            sub_node = ( base, "", [] )
        else:
            sub_node = ( base, name, [] )

        menu[2].append( sub_node )

    if len( rest ) > 0:
        insert_menu_node( sub_node, rest, name )

def get_menu():
    fh = open( os.path.join( cms_dir, "menu.txt" ) )

    menus = []

    for line in fh:
        m = re.match( '^\[(.*)\|(.*)\]$', line )
        if m:
            menus.append( ( m.groups()[0], m.groups()[1] ) )

    root = ( "", "", [] )
    nodes = [ root ]

    for entry, name in menus:
        path = entry.split( "/" )
        insert_menu_node( root, path, name )

    return root[2]

def generate_menu( menu_dir, menu, active_entry, full_active_entry ):

    content = ""

    fh = StringIO()

    path = active_entry.split( "/" )
    base = path[0]
    rest = path[1:]

    parent_files = []

    if full_active_entry.endswith( "/" ):
        base_url = "../" * ( full_active_entry.count( "/" ) - 1 )
    else:
        base_url = "../" * full_active_entry.count( "/" )

    if menu_dir == "":
        print >> fh, "<ul class=\"toplevelmenu\">"
    else:
        print >> fh, "<ul>"

    for menu_entry in menu:
        print >> fh, "<li>"

        if menu_entry[1] == "":
            visible_name = menu_entry[2][0][1]
            entry_path = os.path.join( menu_dir, menu_entry[0], menu_entry[2][0][0] )
            sub_menu = menu_entry[2][0][2] + menu_entry[2][1:]
        else:
            visible_name = menu_entry[1]
            entry_path = os.path.join( menu_dir, menu_entry[0] )
            sub_menu = menu_entry[2]

        if entry_path == full_active_entry:
            print >> fh, "<span class=\"menunona\">", visible_name, "</span>"
        else:
            link = os.path.join( base_url, entry_path + ".html" )
            print >> fh, "<a href=\"%s\">" % ( link ), visible_name, "</a>"

        if base != menu_entry[0]:
            # skip other sub menus not matching the active entry
            sub_menu = []

        if base == menu_entry[0]:
            parent_files.append( entry_path )

        if len( sub_menu ) > 0:
            ( res, sub_content ) = generate_menu( os.path.join( menu_dir, menu_entry[0] ), sub_menu, "/".join( rest ), full_active_entry )
            print >> fh, sub_content

            parent_files += res

        print >> fh, "</li>"

    print >> fh, "</ul>"

    if len( parent_files ) < 1:
        # no entry found but menu has been printed for that base dir
        # so add the active entry
        parent_files.append( full_active_entry )

    return ( parent_files, fh.getvalue() )

def read_file_content( filename ):

    fh = open( os.path.join( cms_dir, filename + ".txt" ) )

    content = []
    tags = {}

    in_tags = True

    line_counter = 0

    for line in fh:

        line = line.rstrip( "\n" )

        line_counter += 1

        if in_tags:
            m = re.match( "^([a-z]+):(.*)$", line )
            if m:
                tags[m.groups()[0]] = m.groups()[1].lstrip( " \t" )
                continue
            else:
                if line == "":
                    in_tags = False
                elif line_counter == 1:
                    in_tags = False
                else:
                    continue

        content.append( line )

    return ( content, tags )

def read_file_content_list( parent_files ):

    ( my_content, my_tags ) = read_file_content( parent_files[0] )

    if len( parent_files ) > 1:
        ( other_content, other_tags ) = read_file_content_list( parent_files[1:] )

        for k in my_tags:
            if not k in other_tags:
                other_tags[k] = my_tags[k]

        return ( other_content, other_tags )
    else:
        return ( my_content, my_tags )

def get_and_copy_file( basedir, current_menu_entry, filename, attr = None ):

    copy_file = True

    if attr is not None:
        for a in attr:
            if a == "copy=0":
                copy_file = False

    if current_menu_entry.endswith( "/" ):
        base_url = "../" * ( current_menu_entry.count( "/" ) - 1 )
    else:
        base_url = "../" * current_menu_entry.count( "/" )

    local_fullname = os.path.normpath( os.path.join( files_dir, filename ) )
    target_filename = os.path.normpath( os.path.join( output_dir, filename ) )
    target_urlname = os.path.normpath( os.path.join( base_url, filename ) )

    if os.path.exists( local_fullname ):
        if copy_file == True and not os.path.exists( target_filename ):
            try:
                os.makedirs( os.path.dirname( target_filename ) )
            except:
                pass
            shutil.copyfile( local_fullname, target_filename )

        size = os.path.getsize( local_fullname )

        return ( True, target_urlname, size, local_fullname, target_filename )
    else:
        return ( False, target_urlname, 0, local_fullname, target_filename )

def get_relative_root_dir( entry ):
    if entry.endswith( "/" ):
        base_url = "../" * ( entry.count( "/" ) - 1 )
    else:
        base_url = "../" * entry.count( "/" )
    if len( base_url ) < 1:
        return "./"
    return base_url

def get_full_filename( filename, current_menu_entry ):
    if filename.startswith( "/" ):
        #full_filename = os.path.join( get_relative_root_dir( current_menu_entry ),
        #                              filename[1:] )
        full_filename = filename[1:]
    else:
        full_filename = os.path.join( os.path.dirname( current_menu_entry ),
                                      filename )
    return full_filename

def get_img_tag( basedir, current_menu_entry, img_file, attr, alt_file = None ):
    # return img tag for img_file, if not found use alt_file to create a thumbnail
    full_imgfile = get_full_filename( img_file, current_menu_entry )

    img_urlname = get_and_copy_file( basedir, current_menu_entry, full_imgfile, attr )
    if img_urlname[0] == False:
        infile = alt_file
        outfile = img_urlname[4]

        if os.path.exists( outfile ):
            print >> sys.stderr, "Reusing thumbnail for", infile
        else:
            img_width = 128
            if attr is not None:
                for a in attr:
                    if a.startswith( "width=" ):
                        img_width = int( a[6:] )

            cmd = [ "convert",
                    infile,
                    "-scale", "%d" % ( img_width ),
                    outfile ]

            print >> sys.stderr, "Creating thumbnail for", infile

            subprocess.call( cmd )

        img_urlname = [ True ] + list( img_urlname[1:] )

    if os.path.exists( img_urlname[4] ):
        cmd = [ "file", img_urlname[4] ]

        res = subprocess.Popen( cmd, stdout = subprocess.PIPE )
        ( out, err ) = res.communicate()

        m = re.match( ".*image.*, ([0-9]+) x ([0-9]+),.*", out )
        if m:
            img_width = int( m.groups()[0] )
            img_height = int( m.groups()[1] )
        else:
            cmd = [ "xli", "-identify", img_urlname[4] ]

            res = subprocess.Popen( cmd, stdout = subprocess.PIPE )
            ( out, err ) = res.communicate()

            m = re.match( ".* is a ([0-9]+)x([0-9]+).*", out )
            if m:
                img_width = int( m.groups()[0] )
                img_height = int( m.groups()[1] )
                print >> sys.stderr, "Got jpg size %d,%d" % ( img_width, img_height)
            else:
                img_width = 80
                img_height = 64
    else:
        img_width = 80
        img_height = 64

    if img_urlname[0] == True:

        img_attrs = ""

        if attr is not None:
            for a in attr:
                if a == "align=right":
                    img_attrs = "align=\"right\""

        link_text = "<img src=\"%s\" width=\"%d\" height=\"%d\" border=\"0\" %s/>" % ( img_urlname[1],
                                                                                       img_width,
                                                                                       img_height,
                                                                                       img_attrs )
    else:
        link_text = None

    return link_text

def replace_link( basedir, current_menu_entry, link ):
    a = link.split( "|" )

    if a[0].startswith( "img:" ):
        target = a[0]

        if len( a ) > 1:
            attr = a[1].split( ":" )
        else:
            attr = []
    else:
        if len( a ) < 2:
            target = a[0]
            title = a[0]
        else:
            target = a[0]
            title = a[1]

        if len( a ) > 2:
            attr = a[2].split( ":" )
        else:
            attr = []

    link_str = ""

    a_attrs = ""

    if attr is not None:
        for a in attr:
            if a.startswith( "name=" ):
                a_attrs = "name=\"%s\"" % ( a[5:] )

    if target.startswith( "http:" ) or target.startswith( "mailto:" ) or target.startswith( "ftp:" ):
        if title.startswith( "img:" ):
            link_text = get_img_tag( basedir, current_menu_entry, title[4:], attr )
            if link_text is None:
                link_text = title
        else:
            link_text = title

        link_str = "<a href=\""
        link_str += target
        link_str += "\" %s>" % ( a_attrs )
        link_str += link_text
        link_str += "</a>"
    elif target.startswith( "file:" ):
        filename = target[5:]
        if len( a ) < 2:
            title = filename

        full_filename = get_full_filename( filename, current_menu_entry )

        target_urlname = get_and_copy_file( basedir, current_menu_entry, full_filename, attr )
        target_signame = get_and_copy_file( basedir, current_menu_entry, full_filename + ".asc", attr )

        if target_urlname[0] == True:

            if title.startswith( "img:" ):
                img_file = title[4:]

                link_text = get_img_tag( basedir, current_menu_entry, img_file, attr, target_urlname[4] )

            else:
                link_text = title
                link_text += " (%s kB)" % ( target_urlname[2] / 1024 )

            link_str = "<a href=\""
            link_str += target_urlname[1]
            link_str += "\" %s>" % ( a_attrs )
            link_str += link_text
            link_str += "</a>"

            if target_signame[0] == True:
                link_str += "<a href=\""
                link_str += target_signame[1]
                link_str += "\">"
                link_str += " (signature)"
                link_str += "</a>"
        else:
            print >> sys.stderr, "File not found for link", target
    elif target.startswith( "img:" ):
        filename = target[4:]

        link_text = get_img_tag( basedir, current_menu_entry, filename, attr )

        if link_text is not None:

            link_str = link_text
        else:
            print >> sys.stderr, "File not found for link", target
    elif target.startswith( "res:" ):
        full_target = target[4:]

        if title.startswith( "img:" ):
            link_text = get_img_tag( basedir, current_menu_entry, title[4:], attr )
            if link_text is None:
                link_text = title
        else:
            link_text = title

        link_str = "<a href=\""
        link_str += full_target
        link_str += "\" %s>" % ( a_attrs )
        link_str += link_text
        link_str += "</a>"
    else:
        if target.startswith( "/" ):
            full_target = os.path.join( get_relative_root_dir( current_menu_entry ),
                                        target[1:] )
        else:
            full_target = target

        if title.startswith( "img:" ):
            link_text = get_img_tag( basedir, current_menu_entry, title[4:], attr )
            if link_text is None:
                link_text = title
        else:
            link_text = title

        link_str = "<a href=\""
        link_str += full_target
        link_str += ".html"
        link_str += "\" %s>" % ( a_attrs )
        link_str += link_text
        link_str += "</a>"

    return link_str

def replace_links( basedir, current_menu_entry, line ):
    in_link = False
    current_link_str = ""
    current_line_str = ""
    next_escaped = False

    for char in line:
        if next_escaped:
            current_line_str += char
            next_escaped = False
        elif char == '\\':
            next_escaped = True
        elif char == '[':
            in_link = True
            current_link_str = ""
        elif char == ']' and in_link == True:
            in_link = False
            current_line_str += replace_link( basedir, current_menu_entry, current_link_str )
        else:
            if in_link:
                current_link_str += char
            else:
                current_line_str += char

    return current_line_str

def transform_file_content( basedir, current_menu_entry, content ):
    new_content = []

    current_list_stack = []

    for line in content:

        h = re.match( "^=(={1,6}) (.*)$", line )
        ol = re.match( "^((?:#|\.)+) (.*)$", line )
        ul = re.match( "^((?:-|\*)+) (.*)$", line )
        ul2 = re.match( "^( +)(?:-|\*) (.*)$", line )
        ol2 = re.match( "^( +)(?:#|\.) (.*)$", line )

        rawinc = re.match( "^ *\[includeraw:(.*)\]$", line )

        if h:
            new_line = "<h%d>%s</h%d>" % ( len( h.groups()[0] ),
                                           h.groups()[1],
                                           len( h.groups()[0] ) )
        elif ul or ol or ul2 or ol2:
            if ul:
                depth = len( ul.groups()[0] )
                new_line = ul.groups()[1]
                list_type = "ul"
            elif ol:
                depth = len( ol.groups()[0] )
                new_line = ol.groups()[1]
                list_type = "ol"
            elif ol2:
                depth = len( ol2.groups()[0] )
                new_line = ol2.groups()[1]
                list_type = "ol"
            else:
                depth = len( ul2.groups()[0] )
                new_line = ul2.groups()[1]
                list_type = "ul"

            if len( current_list_stack ) < 1:
                if depth == 1:
                    # new initial list

                    current_list_stack.append( ( depth, list_type ) )

                    new_line = "<%s><li>" % ( list_type ) + new_line
            else:
                if depth == current_list_stack[-1][0]:
                    # next entry of same depth
                    new_line = "</li><li>" + new_line
                elif ( ( ol or ul ) and depth == current_list_stack[-1][0] + 1 ) or \
                        ( ( ol2 or ul2 ) and depth >= current_list_stack[-1][0] + 2 ):
                    # new list

                    current_list_stack.append( ( depth, list_type ) )

                    new_line = "<%s><li>" % ( list_type ) + new_line
                elif depth < current_list_stack[-1][0]:
                    # closed list and new entry of corresponding depth

                    closing = ""
                    while depth < current_list_stack[-1][0]:
                        closing += "</li></%s>" % ( current_list_stack[-1][1] )
                        current_list_stack = current_list_stack[:-1]
                    new_line = closing + "</li><li>" + new_line
        elif len( line ) < 1 and len( current_list_stack ) > 0:
            # closed list

            closing = ""
            while len( current_list_stack ) > 0:
                closing += "</li></%s>" % ( current_list_stack[-1][1] )
                current_list_stack = current_list_stack[:-1]
            new_line = closing

            current_list_stack = []
        elif rawinc:
            rawfile = rawinc.groups()[0]
            if os.path.exists( rawfile ):
                fh = open( rawfile )
                for rawline in fh:
                    new_content.append( rawline )
                continue
        else:
            new_line = line

        new_line = replace_links( basedir, current_menu_entry, new_line )
        new_content.append( new_line )

    return new_content

def generate_html( basedir, filename, menus ):
    base_without_cms = basedir[ len( cms_dir ):]
    if len( base_without_cms ) > 0 and base_without_cms[0] == "/":
        base_without_cms = base_without_cms[1:]

    target_dir = os.path.join( output_dir, base_without_cms )
    target_file = os.path.join( target_dir, filename[:-4] + ".html" )

    try:
        os.makedirs( target_dir )
    except:
        pass

    current_menu_entry = os.path.join( basedir, filename[:-4] )[len(cms_dir)+1:]

    if current_menu_entry.endswith( "/" ):
        base_url = "../" * ( current_menu_entry.count( "/" ) - 1 )
    else:
        base_url = "../" * current_menu_entry.count( "/" )

    ( parent_files, menu_content ) = generate_menu( "", menus, current_menu_entry, current_menu_entry )

    ( file_content, tags ) = read_file_content_list( parent_files )

    file_content = transform_file_content( basedir, current_menu_entry, file_content )

    fh = open( target_file, "w" )
    
    if "title" in tags:
        title = tags["title"]
    else:
        title = "N/A"

    file_content_str = ""
    for line in file_content:
        file_content_str += line + "\n"

    replace_keywords = { "title" : title,
                         "css_name" : base_url + base_css,
                         "menu_content" : menu_content,
                         "file_content" : file_content_str }

    tpl_fh = open( tpl_file, "r" )
    tpl_content = tpl_fh.read()

    print >> fh, tpl_content.format( **replace_keywords )

def print_version( option, opt_str, value, parser ):
    print "mmu2html.py %s" % ( MMU2HTML_VERSION )
    sys.exit( 0 )

if __name__ == "__main__":
    parser = OptionParser( usage = "usage: %prog [options] [<files>]" )
    parser.add_option( "-c", "--config",
                       action  = "store",
                       type    = "string",
                       dest    = "config",
                       help    = "config" )
    parser.add_option( "-V", "--version",
                       action  = "callback",
                       callback = print_version,
                       help    = "print mmu2html version" )


    ( options, args ) = parser.parse_args()

    config = ConfigParser.ConfigParser()
    config.read( options.config )

    try:
        cms_dir = config.get( "base", "input" )
    except:
        pass
    try:
        files_dir = config.get( "base", "files" )
    except:
        pass
    try:
        output_dir = config.get( "base", "output" )
    except:
        pass
    try:
        base_css = config.get( "base", "css" )
    except:
        pass
    try:
        tpl_file = config.get( "base", "template" )
    except:
        pass

    files_to_copy = []
    try:
        files_to_copy = eval( config.get( "base", "copy" ) )
    except:
        pass

    for f in files_to_copy:
        source = os.path.join( files_dir, f )
        if os.path.exists( source ):
            target = os.path.join( output_dir, f )
            try:
                os.makedirs( os.path.dirname( target ) )
            except:
                pass
            if os.path.isfile( source ):
                shutil.copyfile( source, target )
            elif os.path.isdir( source ):
                if os.path.exists( target ):
                    if os.path.realpath( target ).startswith( os.path.realpath( output_dir ) ):
                        print "Removing existing dir", target
                        shutil.rmtree( target )
                shutil.copytree( source, target )

    menus = get_menu()

    for base, dirs, files in os.walk( cms_dir ):
        for file in files:
            if file != "menu.txt" and file.endswith( ".txt" ):
                generate_html( base, file, menus )
