#! /usr/bin/env python
# mmu2html.py
# Copyright (C) 2013-2014 Ralf Hoffmann.
# ralf@boomerangsworld.de
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import re
import os
import sys
import shutil
import subprocess
from cStringIO import StringIO
import ConfigParser
from optparse import OptionParser

MMU2HTML_VERSION = 0.3

class MMU2Html:

    def __init__( self, config ):
        self.__config = config
        self.__delete_stale = False

        try:
            self.__cms_dir = config.get( "base", "input" )
        except:
            self.__cms_dir = "cms"

        try:
            self.__files_dir = config.get( "base", "files" )
        except:
            self.__files_dir = "files"

        try:
            self.__output_dir = config.get( "base", "output" )
        except:
            self.__output_dir = "html"

        try:
            self.__base_css = config.get( "base", "css" )
        except:
            self.__base_css = "style.css"

        try:
            self.__tpl_file = config.get( "base", "template" )
        except:
            self.__tpl_file = ""

        self.__files_to_copy = []
        try:
            self.__files_to_copy = eval( config.get( "base", "copy" ) )
        except:
            pass

        try:
            self.__char_replace = eval( config.get( "base", "char_replace" ) )
        except:
            self.__char_replace = {}

        self.__old_output_entries = []

        for base, dirs, files in os.walk( self.__output_dir ):
            for filename in files:
                self.__old_output_entries.append( os.path.join( base, filename ) )
            #for dirname in dirs:
            #    self.__old_output_entries.append( os.path.join( base, dirname ) )

    def set_delete_stale( self, value ):
        self.__delete_stale = value

    def generate( self ):
        for f in self.__files_to_copy:
            source = os.path.join( self.__files_dir, f )
            if os.path.exists( source ):
                target = os.path.join( self.__output_dir, f )
                try:
                    os.makedirs( os.path.dirname( target ) )
                except:
                    pass
                if os.path.isfile( source ):
                    shutil.copyfile( source, target )
                    self.remove_from_list( target )
                elif os.path.isdir( source ):
                    if os.path.exists( target ):
                        if os.path.realpath( target ).startswith( os.path.realpath( self.__output_dir ) ):
                            print "Removing existing dir", target
                            shutil.rmtree( target )
                            self.remove_from_list( target )
                    shutil.copytree( source, target )

        menus = self.get_menu()

        for base, dirs, files in os.walk( self.__cms_dir ):
            for file in files:
                if file != "menu.txt" and file.endswith( ".txt" ):
                    self.generate_html( base, file, menus )

        if len( self.__old_output_entries ) > 0:
            for f in self.__old_output_entries:
                if self.__delete_stale:
                    print >> sys.stderr, "Stale file %s in output directory found, removing..." % ( f )
                    os.unlink( f )                    
                else:
                    print >> sys.stderr, "Stale file %s in output directory found." % ( f )

    def insert_menu_node( self, menu, entry, name):
        base = entry[0]
        rest = entry[1:]

        sub_node = None

        for children in menu[2]:
            if children[0] == base:
                sub_node = children
                break

        if sub_node is None:
            if len( rest ) > 0:
                sub_node = ( base, "", [] )
            else:
                sub_node = ( base, name, [] )

            menu[2].append( sub_node )

        if len( rest ) > 0:
            self.insert_menu_node( sub_node, rest, name )

    def get_menu( self ):
        fh = open( os.path.join( self.__cms_dir, "menu.txt" ) )

        menus = []

        for line in fh:
            m = re.match( '^\[(.*)\|(.*)\]$', line )
            if m:
                menus.append( ( m.groups()[0], m.groups()[1] ) )

        root = ( "", "", [] )
        nodes = [ root ]

        for entry, name in menus:
            path = entry.split( "/" )
            self.insert_menu_node( root, path, name )

        return root[2]

    def generate_menu( self, menu_dir, menu, active_entry, full_active_entry ):

        content = ""

        fh = StringIO()

        path = active_entry.split( "/" )
        base = path[0]
        rest = path[1:]

        parent_files = []

        if full_active_entry.endswith( "/" ):
            base_url = "../" * ( full_active_entry.count( "/" ) - 1 )
        else:
            base_url = "../" * full_active_entry.count( "/" )

        if menu_dir == "":
            print >> fh, "<ul class=\"toplevelmenu\">"
        else:
            print >> fh, "<ul>"

        for menu_entry in menu:
            print >> fh, "<li>"

            if menu_entry[1] == "":
                visible_name = menu_entry[2][0][1]
                entry_path = os.path.join( menu_dir, menu_entry[0], menu_entry[2][0][0] )
                sub_menu = menu_entry[2][0][2] + menu_entry[2][1:]
            else:
                visible_name = menu_entry[1]
                entry_path = os.path.join( menu_dir, menu_entry[0] )
                sub_menu = menu_entry[2]

            if entry_path == full_active_entry:
                print >> fh, "<span class=\"menunona\">", visible_name, "</span>"
            else:
                link = os.path.join( base_url, entry_path + ".html" )
                print >> fh, "<a href=\"%s\">" % ( link ), visible_name, "</a>"

            if base != menu_entry[0]:
                # skip other sub menus not matching the active entry
                sub_menu = []

            if base == menu_entry[0]:
                parent_files.append( entry_path )

            if len( sub_menu ) > 0:
                ( res, sub_content ) = self.generate_menu( os.path.join( menu_dir, menu_entry[0] ), sub_menu, "/".join( rest ), full_active_entry )
                print >> fh, sub_content

                parent_files += res

            print >> fh, "</li>"

        print >> fh, "</ul>"

        if len( parent_files ) < 1:
            # no entry found but menu has been printed for that base dir
            # so add the active entry
            parent_files.append( full_active_entry )

        return ( parent_files, fh.getvalue() )

    def read_file_content( self, filename ):

        fh = open( os.path.join( self.__cms_dir, filename + ".txt" ) )

        content = []
        tags = {}

        in_tags = True

        line_counter = 0

        for line in fh:

            line = line.rstrip( "\n" )

            line_counter += 1

            if in_tags:
                m = re.match( "^([a-z_]+):(.*)$", line )
                if m:
                    tags[m.groups()[0]] = m.groups()[1].lstrip( " \t" )
                    continue
                else:
                    if line == "":
                        in_tags = False
                    elif line_counter == 1:
                        in_tags = False
                    else:
                        continue

            content.append( line )

        return ( content, tags )

    def read_file_content_list( self, parent_files ):

        ( my_content, my_tags ) = self.read_file_content( parent_files[0] )

        if len( parent_files ) > 1:
            ( other_content, other_tags ) = self.read_file_content_list( parent_files[1:] )

            for k in my_tags:
                if not k in other_tags:
                    other_tags[k] = my_tags[k]

            return ( other_content, other_tags )
        else:
            return ( my_content, my_tags )

    def get_and_copy_file( self, basedir, current_menu_entry, filename, attr = None ):

        copy_file = True

        if attr is not None:
            for a in attr:
                if a == "copy=0":
                    copy_file = False

        if current_menu_entry.endswith( "/" ):
            base_url = "../" * ( current_menu_entry.count( "/" ) - 1 )
        else:
            base_url = "../" * current_menu_entry.count( "/" )

        local_fullname = os.path.normpath( os.path.join( self.__files_dir, filename ) )
        target_filename = os.path.normpath( os.path.join( self.__output_dir, filename ) )
        target_urlname = os.path.normpath( os.path.join( base_url, filename ) )

        if os.path.exists( local_fullname ):
            if copy_file == True and not os.path.exists( target_filename ):
                try:
                    os.makedirs( os.path.dirname( target_filename ) )
                except:
                    pass
                shutil.copyfile( local_fullname, target_filename )

            self.remove_from_list( target_filename )

            size = os.path.getsize( local_fullname )

            return ( True, target_urlname, size, local_fullname, target_filename )
        else:
            return ( False, target_urlname, 0, local_fullname, target_filename )

    def get_relative_root_dir( self, entry ):
        if entry.endswith( "/" ):
            base_url = "../" * ( entry.count( "/" ) - 1 )
        else:
            base_url = "../" * entry.count( "/" )
        if len( base_url ) < 1:
            return "./"
        return base_url

    def get_full_filename( self, filename, current_menu_entry ):
        if filename.startswith( "/" ):
            #full_filename = os.path.join( self.get_relative_root_dir( current_menu_entry ),
            #                              filename[1:] )
            full_filename = filename[1:]
        else:
            full_filename = os.path.join( os.path.dirname( current_menu_entry ),
                                          filename )
        return full_filename

    def get_img_tag( self, basedir, current_menu_entry, img_file, attr, alt_file = None ):
        # return img tag for img_file, if not found use alt_file to create a thumbnail
        full_imgfile = self.get_full_filename( img_file, current_menu_entry )

        img_urlname = self.get_and_copy_file( basedir, current_menu_entry, full_imgfile, attr )
        if img_urlname[0] == False:
            if alt_file is None:
                print >> sys.stderr, "File", img_file, "not found"
                return None

            infile = alt_file
            outfile = img_urlname[4]

            if os.path.exists( outfile ):
                print >> sys.stderr, "Reusing thumbnail for", infile
            else:
                img_width = 128
                if attr is not None:
                    for a in attr:
                        if a.startswith( "width=" ):
                            img_width = int( a[6:] )

                cmd = [ "convert",
                        infile,
                        "-scale", "%d" % ( img_width ),
                        outfile ]

                print >> sys.stderr, "Creating thumbnail for", infile

                subprocess.call( cmd )

            self.remove_from_list( outfile )

            img_urlname = [ True ] + list( img_urlname[1:] )

        if os.path.exists( img_urlname[4] ):
            cmd = [ "file", img_urlname[4] ]

            res = subprocess.Popen( cmd, stdout = subprocess.PIPE )
            ( out, err ) = res.communicate()

            m = re.match( ".*image.*, ([0-9]+) x ([0-9]+),.*", out )
            if m:
                img_width = int( m.groups()[0] )
                img_height = int( m.groups()[1] )
            else:
                cmd = [ "xli", "-identify", img_urlname[4] ]

                res = subprocess.Popen( cmd, stdout = subprocess.PIPE )
                ( out, err ) = res.communicate()

                m = re.match( ".* is a ([0-9]+)x([0-9]+).*", out )
                if m:
                    img_width = int( m.groups()[0] )
                    img_height = int( m.groups()[1] )
                    print >> sys.stderr, "Got jpg size %d,%d" % ( img_width, img_height)
                else:
                    img_width = 80
                    img_height = 64
        else:
            img_width = 80
            img_height = 64

        if img_urlname[0] == True:

            img_attrs = ""

            if attr is not None:
                for a in attr:
                    if a == "align=right":
                        img_attrs = "align=\"right\""

            link_text = "<img src=\"%s\" width=\"%d\" height=\"%d\" border=\"0\" alt=\"\" %s/>" % ( img_urlname[1],
                                                                                                    img_width,
                                                                                                    img_height,
                                                                                                    img_attrs )
        else:
            link_text = None

        return link_text

    def replace_special_chars( self, line ):
        res = ""

        for char in line:
            if char in self.__char_replace:
                res += self.__char_replace[char];
            else:
                res += char

        return res

    def replace_link( self, basedir, current_menu_entry, link ):
        a = link.split( "|" )

        if a[0].startswith( "img:" ):
            target = a[0]

            if len( a ) > 1:
                attr = a[1].split( ":" )
            else:
                attr = []
        else:
            if len( a ) < 2:
                target = a[0]
                title = a[0]
            else:
                target = a[0]
                title = a[1]

            if len( a ) > 2:
                attr = a[2].split( ":" )
            else:
                attr = []

        link_str = ""

        a_attrs = ""

        if attr is not None:
            for a in attr:
                if a.startswith( "name=" ):
                    a_attrs = "name=\"%s\"" % ( a[5:] )

        if target.startswith( "http:" ) or target.startswith( "mailto:" ) or target.startswith( "ftp:" ):
            if title.startswith( "img:" ):
                link_text = self.get_img_tag( basedir, current_menu_entry, title[4:], attr )
                if link_text is None:
                    link_text = title
            else:
                link_text = title

            link_str = "<a href=\""
            link_str += target
            link_str += "\" %s>" % ( a_attrs )
            link_str += link_text
            link_str += "</a>"
        elif target.startswith( "file:" ):
            filename = target[5:]
            if len( a ) < 2:
                title = filename

            full_filename = self.get_full_filename( filename, current_menu_entry )

            target_urlname = self.get_and_copy_file( basedir, current_menu_entry, full_filename, attr )
            target_signame = self.get_and_copy_file( basedir, current_menu_entry, full_filename + ".asc", attr )

            if target_urlname[0] == True:

                if title.startswith( "img:" ):
                    img_file = title[4:]

                    link_text = self.get_img_tag( basedir, current_menu_entry, img_file, attr, target_urlname[4] )

                else:
                    link_text = title
                    link_text += " (%s kB)" % ( target_urlname[2] / 1024 )

                link_str = "<a href=\""
                link_str += target_urlname[1]
                link_str += "\" %s>" % ( a_attrs )
                link_str += link_text
                link_str += "</a>"

                if target_signame[0] == True:
                    link_str += "<a href=\""
                    link_str += target_signame[1]
                    link_str += "\">"
                    link_str += " (signature)"
                    link_str += "</a>"
            else:
                print >> sys.stderr, "File not found for link", target
        elif target.startswith( "img:" ):
            filename = target[4:]

            link_text = self.get_img_tag( basedir, current_menu_entry, filename, attr )

            if link_text is not None:

                link_str = link_text
            else:
                print >> sys.stderr, "File not found for link", target
        elif target.startswith( "res:" ):
            full_target = target[4:]

            if title.startswith( "img:" ):
                link_text = self.get_img_tag( basedir, current_menu_entry, title[4:], attr )
                if link_text is None:
                    link_text = title
            else:
                link_text = title

            link_str = "<a href=\""
            link_str += full_target
            link_str += "\" %s>" % ( a_attrs )
            link_str += link_text
            link_str += "</a>"
        elif len( a ) == 1 and target[0] == '#':
            link_str = "<a name=\""
            link_str += target[1:]
            link_str += "\"></a>"
        else:
            if target.find( "#" ) >= 0:
                actual_target = target[:target.find( "#" )]
                anchor = target[target.find( "#" ) + 1:]
            else:
                actual_target = target
                anchor = ""

            if actual_target.startswith( "/" ):
                full_target = os.path.join( self.get_relative_root_dir( current_menu_entry ),
                                            actual_target[1:] )
            else:
                full_target = actual_target

            if title.startswith( "img:" ):
                link_text = self.get_img_tag( basedir, current_menu_entry, title[4:], attr )
                if link_text is None:
                    link_text = title
            else:
                link_text = title

            link_str = "<a href=\""
            link_str += full_target
            link_str += ".html"

            if len( anchor ) > 0:
                link_str += "#"
                link_str += anchor

            link_str += "\" %s>" % ( a_attrs )
            link_str += link_text
            link_str += "</a>"

        return self.replace_special_chars( link_str )

    def replace_text_formatting( self, line ):
        mode = [ "none" ]
        current_line_str = ""
        next_escaped = False

        for char in line:
            if next_escaped:
                current_line_str += char
                next_escaped = False
            elif char == '\\':
                next_escaped = True
            elif char == '*':
                if mode[-1] != "bold":
                    mode.append( "bold" )
                    current_line_str += "<b>"
                elif mode[-1] == "bold":
                    current_line_str += "</b>"
                    mode = mode[:-1]
            elif char == '_':
                if mode[-1] != "underline":
                    mode.append( "underline" )
                    current_line_str += "<u>"
                elif mode[-1] == "underline":
                    current_line_str += "</u>"
                    mode = mode[:-1]
            else:
                current_line_str += char

        return self.replace_special_chars( current_line_str )

    def replace_links( self, basedir, current_menu_entry, line ):
        in_link = False
        current_link_str = ""
        current_temp_line_str = ""
        current_line_str = ""
        next_escaped = False

        for char in line:
            if next_escaped:
                if char == '[' or char == ']':
                    current_temp_line_str += char
                else:
                    current_temp_line_str += '\\' + char
                next_escaped = False
            elif char == '\\':
                next_escaped = True
            elif char == '[':
                in_link = True
                current_link_str = ""
            elif char == ']' and in_link == True:
                in_link = False
                current_line_str += self.replace_text_formatting( current_temp_line_str )
                current_line_str += self.replace_link( basedir, current_menu_entry, current_link_str )

                current_temp_line_str = ""
            else:
                if in_link:
                    current_link_str += char
                else:
                    current_temp_line_str += char

        current_line_str += self.replace_text_formatting( current_temp_line_str )

        return current_line_str

    def strip_links( self, line ):
        while True:
            m = re.match( "^([^\[]*)\[[^\]]+\](.*)$", line )
            if not m:
                break

            line = m.groups()[0] + m.groups()[1]
        return line

    def transform_file_content( self, basedir, current_menu_entry, content, options ):
        new_content = []

        current_list_stack = []

        header_number = []
        toc = []

        insert_toc_at_pos = -1

        for line in content:

            h = re.match( "^=(={1,6}) (.*)$", line )
            ol = re.match( "^((?:#|\.)+) (.*)$", line )
            ul = re.match( "^((?:-|\*)+) (.*)$", line )
            ul2 = re.match( "^( +)(?:-|\*) (.*)$", line )
            ol2 = re.match( "^( +)(?:#|\.) (.*)$", line )

            rawinc = re.match( "^ *\[includeraw:(.*)\]$", line )
            insert_toc = re.match( "^\[toc\]$", line )

            if h:

                hlevel = len( h.groups()[0] )

                while len( header_number ) < hlevel - 1:
                    header_number.append( 0 )

                if len( header_number ) < hlevel:
                    header_number.append( 1 )
                else:
                    header_number[hlevel - 1] += 1

                header_number = header_number[:hlevel]

                tline = ""
                aname = "h"

                initial_missing = True

                for p in range( hlevel ):
                    if header_number[p] != 0:
                        initial_missing = False

                    if initial_missing == False or header_number[p] != 0:
                        aname += "-%d" % ( header_number[p] )

                if "number_headings" in options and options["number_headings"]:

                    initial_missing = True

                    for p in range( hlevel ):

                        if header_number[p] != 0:
                            initial_missing = False

                        if initial_missing == False or header_number[p] != 0:
                            tline += "%d." % ( header_number[p] )

                    tline += " %s" % ( h.groups()[1] )
                else:
                    tline += "%s" % ( h.groups()[1] )

                new_line = "<h%d><a name=\"%s\"></a>" % ( hlevel, aname )

                new_line += tline

                toc.append( ( self.strip_links( tline ), aname, hlevel, header_number[hlevel - 1] ) )

                new_line += "</h%d>" % ( hlevel )

            elif ul or ol or ul2 or ol2:
                if ul:
                    depth = len( ul.groups()[0] )
                    new_line = ul.groups()[1]
                    list_type = "ul"
                elif ol:
                    depth = len( ol.groups()[0] )
                    new_line = ol.groups()[1]
                    list_type = "ol"
                elif ol2:
                    depth = len( ol2.groups()[0] )
                    new_line = ol2.groups()[1]
                    list_type = "ol"
                else:
                    depth = len( ul2.groups()[0] )
                    new_line = ul2.groups()[1]
                    list_type = "ul"

                if len( current_list_stack ) < 1:
                    if depth == 1:
                        # new initial list

                        current_list_stack.append( ( depth, list_type ) )

                        new_line = "<%s><li>" % ( list_type ) + new_line
                else:
                    if depth == current_list_stack[-1][0]:
                        # next entry of same depth
                        new_line = "</li><li>" + new_line
                    elif ( ( ol or ul ) and depth == current_list_stack[-1][0] + 1 ) or \
                            ( ( ol2 or ul2 ) and depth >= current_list_stack[-1][0] + 2 ):
                        # new list

                        current_list_stack.append( ( depth, list_type ) )

                        new_line = "<%s><li>" % ( list_type ) + new_line
                    elif depth < current_list_stack[-1][0]:
                        # closed list and new entry of corresponding depth

                        closing = ""
                        while depth < current_list_stack[-1][0]:
                            closing += "</li></%s>" % ( current_list_stack[-1][1] )
                            current_list_stack = current_list_stack[:-1]
                        new_line = closing + "</li><li>" + new_line
            elif len( line ) < 1 and len( current_list_stack ) > 0:
                # closed list

                closing = ""
                while len( current_list_stack ) > 0:
                    closing += "</li></%s>" % ( current_list_stack[-1][1] )
                    current_list_stack = current_list_stack[:-1]
                new_line = closing

                current_list_stack = []
            elif rawinc:
                rawfile = rawinc.groups()[0]
                if os.path.exists( rawfile ):
                    fh = open( rawfile )
                    for rawline in fh:
                        new_content.append( rawline )
                    continue
            elif insert_toc:
                insert_toc_at_pos = len( new_content )
            else:
                new_line = line

            new_line = self.replace_links( basedir, current_menu_entry, new_line )

            new_content.append( new_line )

        if insert_toc_at_pos >= 0:
            new_content_with_toc = new_content[0:insert_toc_at_pos]

            current_level = toc[0][2]
            start_level = current_level

            new_line = "<div id=\"toc\"><ul>"
            new_content_with_toc.append( new_line )
            for toc_entry, toc_aname, toc_level, toc_number in toc:

                if toc_level == current_level:

                    if toc_number == 1:
                        new_line = "<li id=\"toc_entry\">"
                    else:
                        new_line = "</li><li id=\"toc_entry\">"

                    new_line += "<a href=\"#%s\">%s</a>" % ( toc_aname, toc_entry )
                elif toc_level > current_level:
                    new_line = ""
                    while current_level < toc_level:
                        new_line += "<ul><li id=\"toc_entry\">"
                        current_level += 1

                    new_line += "<a href=\"#%s\">%s</a>" % ( toc_aname, toc_entry )

                    current_level = toc_level
                else:
                    new_line = ""
                    while current_level > toc_level:
                        new_line += "</li></ul>"
                        current_level -= 1
                    new_line += "</li><li id=\"toc_entry\"><a href=\"#%s\">%s</a>" % ( toc_aname, toc_entry )

                    current_level = toc_level

                new_content_with_toc.append( new_line )

            new_line = ""
            while current_level > ( start_level - 1 ):
                new_line += "</li></ul>"
                current_level -= 1

            new_line += "</div>"
            new_content_with_toc.append( new_line )

            new_content_with_toc += new_content[insert_toc_at_pos + 1:]

            new_content = new_content_with_toc

        return new_content

    def generate_html( self, basedir, filename, menus ):
        base_without_cms = basedir[ len( self.__cms_dir ):]
        if len( base_without_cms ) > 0 and base_without_cms[0] == "/":
            base_without_cms = base_without_cms[1:]

        target_dir = os.path.join( self.__output_dir, base_without_cms )
        target_file = os.path.join( target_dir, filename[:-4] + ".html" )

        try:
            os.makedirs( target_dir )
        except:
            pass

        current_menu_entry = os.path.join( basedir, filename[:-4] )[len(self.__cms_dir)+1:]

        if current_menu_entry.endswith( "/" ):
            base_url = "../" * ( current_menu_entry.count( "/" ) - 1 )
        else:
            base_url = "../" * current_menu_entry.count( "/" )

        ( parent_files, menu_content ) = self.generate_menu( "", menus, current_menu_entry, current_menu_entry )

        ( file_content, tags ) = self.read_file_content_list( parent_files )

        number_headings = False
        if "number_headings" in tags:
            if tags["number_headings"] in ( "yes", "true" ):
                number_headings = True

        options = { "number_headings" : number_headings }

        file_content = self.transform_file_content( basedir, current_menu_entry, file_content, options )

        fh = open( target_file, "w" )

        self.remove_from_list( target_file )

        if "title" in tags:
            title = tags["title"]
        else:
            title = "N/A"

        file_content_str = ""
        for line in file_content:
            file_content_str += line + "\n"

        replace_keywords = { "title" : title,
                             "css_name" : base_url + self.__base_css,
                             "menu_content" : menu_content,
                             "file_content" : file_content_str }

        tpl_fh = open( self.__tpl_file, "r" )
        tpl_content = tpl_fh.read()

        print >> fh, tpl_content.format( **replace_keywords )

    def remove_from_list( self, path ):
        to_remove = []

        for p in self.__old_output_entries:
            if p.startswith( path ):
                to_remove.append( p )

        for p in to_remove:
            self.__old_output_entries.remove( p )

def print_version( option, opt_str, value, parser ):
    print "mmu2html.py %s" % ( MMU2HTML_VERSION )
    sys.exit( 0 )

if __name__ == "__main__":
    parser = OptionParser( usage = "usage: %prog [options] [<files>]" )
    parser.add_option( "-c", "--config",
                       action  = "store",
                       type    = "string",
                       dest    = "config",
                       help    = "config" )
    parser.add_option( "-d", "--delete",
                   action  = "store_true",
                   dest    = "delete_stale",
                   default = False,
                   help    = "delete stale output files" )
    parser.add_option( "-V", "--version",
                       action  = "callback",
                       callback = print_version,
                       help    = "print mmu2html version" )


    ( options, args ) = parser.parse_args()

    config = ConfigParser.ConfigParser()
    config.read( options.config )

    m = MMU2Html( config )

    m.set_delete_stale( options.delete_stale )

    m.generate()
