#! /usr/bin/env python
# mmu2html.py
# Copyright (C) 2013-2019 Ralf Hoffmann.
# ralf@boomerangsworld.de
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import re
import os
import sys
import shutil
import subprocess
from cStringIO import StringIO
import ConfigParser
from optparse import OptionParser
import hashlib
from taginfo import TagInfo

MMU2HTML_VERSION = 0.5

class MMU2Html:

    def __init__( self, config ):
        self.__config = config
        self.__delete_stale = False
        self.__validate_html = False
        self.__verbose = False
        self.__tag_info = {}
        self.__special_files = [ "menu.txt", "group.txt", "by-tags.txt" ]
        self.__blog_entries = []

        try:
            self.__cms_dir = config.get( "base", "input" )
        except:
            self.__cms_dir = "cms"

        try:
            self.__files_dir = config.get( "base", "files" )
        except:
            self.__files_dir = "files"

        try:
            self.__output_dir = config.get( "base", "output" )
        except:
            self.__output_dir = "html"

        try:
            self.__base_css = config.get( "base", "css" )
        except:
            self.__base_css = "style.css"

        try:
            self.__tpl_file = config.get( "base", "template" )
        except:
            self.__tpl_file = ""

        self.__files_to_copy = []
        try:
            self.__files_to_copy = eval( config.get( "base", "copy" ) )
        except:
            pass

        try:
            self.__char_replace = eval( config.get( "base", "char_replace" ) )
        except:
            self.__char_replace = {}

        try:
            self.__cache = config.get( "base", "cache" )
        except:
            self.__cache = "cache"

        try:
            self.__sign_files = bool( int( config.get( "base", "sign" ) ) )
        except:
            self.__sign_files = False

        try:
            self.__sign_key = config.get( "base", "sign_key" )
        except:
            self.__sign_key = None

        try:
            v = config.get( "base", "separate_submenu" )
            self.__separate_submenu = ( eval( v ) == True )
        except:
            self.__separate_submenu = False

        self.__old_output_entries = []

        for base, dirs, files in os.walk( self.__output_dir ):
            for filename in files:
                self.__old_output_entries.append( os.path.join( base, filename ) )
            #for dirname in dirs:
            #    self.__old_output_entries.append( os.path.join( base, dirname ) )

    def set_delete_stale( self, value ):
        self.__delete_stale = value

    def set_validate_html( self, value ):
        self.__validate_html = value

    def set_verbose_output( self, value ):
        self.__verbose = value

    def generate( self ):
        for f in self.__files_to_copy:
            source = os.path.join( self.__files_dir, f )
            if os.path.exists( source ):
                target = os.path.join( self.__output_dir, f )
                try:
                    os.makedirs( os.path.dirname( target ) )
                except:
                    pass
                if os.path.isfile( source ):
                    shutil.copy2( source, target )
                    self.remove_from_list( target )
                elif os.path.isdir( source ):
                    if os.path.exists( target ):
                        if os.path.realpath( target ).startswith( os.path.realpath( self.__output_dir ) ):
                            if self.__verbose:
                                print "Removing existing dir", target
                            shutil.rmtree( target )
                            self.remove_from_list( target )
                    shutil.copytree( source, target )

        menus = self.get_menu()
        groupmenus = self.get_groupmenu()

        for stage in [ 1, 2 ]:
            for base, dirs, files in os.walk( self.__cms_dir ):
                for file in files:
                    if not file in self.__special_files and file.endswith( ".txt" ):
                        self.generate_html( base, file, menus, groupmenus, stage = stage )

            if stage == 1:
                self.__blog_entries.sort( key = lambda e: e[1], reverse = True )

        if os.path.exists( os.path.join( self.__cms_dir, "by-tags" ) + ".txt" ):
            for tag in self.__tag_info.keys():
                replace_keywords = { "tag" : tag }
                self.generate_html( os.path.join( self.__cms_dir, "by-tags" ),
                                    tag + ".txt",
                                    menus,
                                    groupmenus,
                                    template = "by-tags",
                                    template_replacements = replace_keywords )

        if len( self.__old_output_entries ) > 0:
            for f in self.__old_output_entries:
                if self.__delete_stale:
                    if self.__verbose:
                        print >> sys.stderr, "Stale file %s in output directory found, removing..." % ( f )
                    os.unlink( f )                    
                else:
                    if self.__verbose:
                        print >> sys.stderr, "Stale file %s in output directory found." % ( f )

    def insert_menu_node( self, menu, entry, name):
        base = entry[0]
        rest = entry[1:]

        sub_node = None

        for children in menu[2]:
            if children[0] == base:
                sub_node = children
                break

        if sub_node is None:
            if len( rest ) > 0:
                sub_node = ( base, "", [] )
            else:
                sub_node = ( base, name, [] )

            menu[2].append( sub_node )

        if len( rest ) > 0:
            self.insert_menu_node( sub_node, rest, name )

    def get_menu( self ):
        fh = open( os.path.join( self.__cms_dir, "menu.txt" ) )

        menus = []

        for line in fh:
            m = re.match( '^\[(.*)\|(.*)\]$', line )
            if m:
                menus.append( ( m.groups()[0], m.groups()[1] ) )

        root = ( "", "", [] )
        nodes = [ root ]

        for entry, name in menus:
            path = entry.split( "/" )
            self.insert_menu_node( root, path, name )

        return root[2]

    def get_groupmenu( self ):
        if not os.path.exists( os.path.join( self.__cms_dir, "group.txt" ) ):
            return [ "", "", {} ]

        fh = open( os.path.join( self.__cms_dir, "group.txt" ) )

        groupmenus = []

        for line in fh:
            m = re.match( '^\[([^|]*)\|([^|]*)\|([^|]*)(\|([^|]*))?\]$', line )
            if m:
                groupmenus.append( ( m.groups()[0], m.groups()[1], m.groups()[2], m.groups()[4] ) )

        groups = [ "", "", {} ]

        for entry, link, name, pos in groupmenus:
            if len( entry ) == 0:
                groups[0] = name
                groups[1] = link
            else:
                if name not in groups[2]:
                    groups[2][name] = ( link, [ entry ], pos )
                else:
                    groups[2][name][1].append( entry )

        return groups

    def generate_menu( self, menu_dir, menu, active_entry, full_active_entry, groupmenus ):

        content = ""

        fh = StringIO()
        fh_toplevel = StringIO()
        fh_submenu = StringIO()

        path = active_entry.split( "/" )
        base = path[0]
        rest = path[1:]

        parent_files = []

        if full_active_entry.endswith( "/" ):
            base_url = "../" * ( full_active_entry.count( "/" ) - 1 )
        else:
            base_url = "../" * full_active_entry.count( "/" )

        if menu_dir == "":
            fh2 = fh_toplevel
        else:
            fh2 = fh_submenu

        if menu_dir == "":
            print >> fh, "<ul class=\"toplevelmenu\">"
            print >> fh2, "<ul class=\"toplevelmenu\">"
        else:
            print >> fh, "<ul class=\"submenu\">"
            print >> fh2, "<ul class=\"submenu\">"

        if menu_dir == "":
            active_group = self.get_group( full_active_entry, groupmenus )
        else:
            #active_group = None
            active_group = self.get_group( full_active_entry, groupmenus )

        for menu_entry in menu:
            if menu_entry[1] == "":
                visible_name = menu_entry[2][0][1]
                entry_path = os.path.join( menu_dir, menu_entry[0], menu_entry[2][0][0] )

                if menu_dir == "" and self.__separate_submenu == True:
                    sub_menu = menu_entry[2][0][2] + menu_entry[2][0:]
                else:
                    sub_menu = menu_entry[2][0][2] + menu_entry[2][1:]
            else:
                visible_name = menu_entry[1]
                entry_path = os.path.join( menu_dir, menu_entry[0] )
                sub_menu = menu_entry[2]

            if self.get_group( entry_path, groupmenus ) != active_group:
                continue

            print >> fh, "<li>"
            print >> fh2, "<li>"

            if entry_path == full_active_entry:
                nona = True
            else:
                nona = False

            if menu_dir == "" and self.__separate_submenu == True:
                if full_active_entry.startswith( menu_entry[0] + "/" ):
                    nona = True

            if nona == True:
                print >> fh, "<span class=\"menunona\">", visible_name, "</span>"
                print >> fh2, "<span class=\"menunona\">", visible_name, "</span>"
            else:
                link = os.path.join( base_url, entry_path + ".html" )
                print >> fh, "<a href=\"%s\">" % ( link ), visible_name, "</a>"
                print >> fh2, "<a href=\"%s\">" % ( link ), visible_name, "</a>"

            if base != menu_entry[0]:
                # skip other sub menus not matching the active entry
                sub_menu = []

            if base == menu_entry[0]:
                if not entry_path in parent_files:
                    parent_files.append( entry_path )

            if len( sub_menu ) > 0:
                ( res, sub_content, sub_toplevel, sub_submenu ) = self.generate_menu( os.path.join( menu_dir, menu_entry[0] ), sub_menu, "/".join( rest ), full_active_entry, groupmenus )
                print >> fh, sub_content
                if menu_dir == "":
                    print >> fh_submenu, sub_content

                for sub_pf in res:
                    if sub_pf not in parent_files:
                        parent_files.append( sub_pf )

            print >> fh, "</li>"
            print >> fh2, "</li>"

        print >> fh, "</ul>"
        print >> fh2, "</ul>"

        if not full_active_entry in parent_files:
            parent_files.append( full_active_entry )

        return ( parent_files, fh.getvalue(), fh_toplevel.getvalue(), fh_submenu.getvalue() )

    def __sort_group( self, g1, g2 ):
        if g1[1] is None and g2[1] is None:
            if g1[0] < g2[0]:
                return -1
            if g1[0] > g2[0]:
                return 1
            return 0
        if g1[1] is None:
            return 1
        if g2[1] is None:
            return -1
        return int( g1[1] ) - int( g2[1] )

    def generate_groupmenu( self, current_menu_entry, groupmenus, full_active_entry ):
        if full_active_entry.endswith( "/" ):
            base_url = "../" * ( full_active_entry.count( "/" ) - 1 )
        else:
            base_url = "../" * full_active_entry.count( "/" )

        content = ""

        fh = StringIO()

        found = False

        sorted_groups = []

        for menu_entry in groupmenus[2].keys():
            sorted_groups.append( ( menu_entry, groupmenus[2][menu_entry][2] ) )

        sorted_groups.sort( cmp=self.__sort_group )

        for menu_entry, pos in sorted_groups:
            print >> fh, "<li>"

            nona = False

            for prefix in groupmenus[2][menu_entry][1]:
                if current_menu_entry.startswith( prefix ):
                    found = True
                    nona = True

            if nona == True:
                print >> fh, "<span class=\"menunona\">", menu_entry, "</span>"
            else:
                link = os.path.join( base_url, groupmenus[2][menu_entry][0] + ".html" )
                print >> fh, "<a href=\"%s\">" % ( link ), menu_entry, "</a>"

            print >> fh, "</li>"

        content = fh.getvalue()

        fh = StringIO()

        print >> fh, "<ul class=\"groupmenu\">"

        print >> fh, "<li>"
        if not found:
            print >> fh, "<span class=\"menunona\">", groupmenus[0], "</span>"
        else:
            link = os.path.join( base_url, groupmenus[1] + ".html" )
            print >> fh, "<a href=\"%s\">" % ( link ), groupmenus[0], "</a>"
        print >> fh, "</li>"

        print >> fh, content
        print >> fh, "</ul>"

        return fh.getvalue()

    def get_group( self, current_menu_entry, groupmenus ):
        group = None

        if not groupmenus:
            return None

        for menu_entry in groupmenus[2].keys():
            for prefix in groupmenus[2][menu_entry][1]:
                if current_menu_entry.startswith( prefix ):
                    group = menu_entry
                    break
            if group:
                break

        if group:
            return group

        return groupmenus[0]

    def read_file_content( self, filename ):

        fh = open( os.path.join( self.__cms_dir, filename + ".txt" ) )

        content = []
        tags = {}

        in_tags = True

        line_counter = 0

        for line in fh:

            line = line.rstrip( "\n" )

            line_counter += 1

            if in_tags:
                m = re.match( "^([a-z_\-]+):(.*)$", line )
                if m:
                    tags[m.groups()[0]] = m.groups()[1].lstrip( " \t" )
                    continue
                else:
                    if line == "":
                        in_tags = False
                    elif line_counter == 1:
                        in_tags = False
                    else:
                        continue

            content.append( line )

        return ( content, tags )

    def read_file_content_list( self, parent_files ):

        ( my_content, my_tags ) = self.read_file_content( parent_files[0] )

        if len( parent_files ) > 1:
            ( my_content, other_tags ) = self.read_file_content_list( parent_files[1:] )

            for k in other_tags:
                my_tags[k] = other_tags[k]

        return ( my_content, my_tags )

    def get_and_copy_file( self, current_menu_entry, filename, attr = None, sign = False ):

        copy_file = True

        if attr is not None:
            for a in attr:
                if a == "copy=0":
                    copy_file = False

        if current_menu_entry.endswith( "/" ):
            base_url = "../" * ( current_menu_entry.count( "/" ) - 1 )
        else:
            base_url = "../" * current_menu_entry.count( "/" )

        local_fullname = os.path.normpath( os.path.join( self.__files_dir, filename ) )
        target_filename = os.path.normpath( os.path.join( self.__output_dir, filename ) )
        target_urlname = os.path.normpath( os.path.join( base_url, filename ) )

        if os.path.exists( local_fullname ):
            if copy_file == True and os.path.exists( target_filename ):
                if os.path.getmtime( local_fullname ) <= os.path.getmtime( target_filename ):
                    copy_file = False

            if copy_file == True:
                try:
                    os.makedirs( os.path.dirname( target_filename ) )
                except:
                    pass
                shutil.copy2( local_fullname, target_filename )

            if sign:
                self.sign_file( target_filename )
                
            self.remove_from_list( target_filename )

            size = os.path.getsize( local_fullname )

            return ( True, target_urlname, size, local_fullname, target_filename )
        else:
            return ( False, target_urlname, 0, local_fullname, target_filename )

    def get_relative_root_dir( self, entry ):
        if entry.endswith( "/" ):
            base_url = "../" * ( entry.count( "/" ) - 1 )
        else:
            base_url = "../" * entry.count( "/" )
        if len( base_url ) < 1:
            return "./"
        return base_url

    def get_full_filename( self, filename, current_menu_entry ):
        if filename.startswith( "/" ):
            #full_filename = os.path.join( self.get_relative_root_dir( current_menu_entry ),
            #                              filename[1:] )
            full_filename = filename[1:]
        else:
            full_filename = os.path.join( os.path.dirname( current_menu_entry ),
                                          filename )
        return full_filename

    def get_img_tag( self, current_menu_entry, img_file, attr, alt_file = None, target_menu_entry = None ):
        use_menu_entry = current_menu_entry
        if target_menu_entry:
            use_menu_entry = target_menu_entry

        # return img tag for img_file, if not found use alt_file to create a thumbnail
        full_imgfile = self.get_full_filename( img_file, current_menu_entry )

        img_urlname = self.get_and_copy_file( use_menu_entry, full_imgfile, attr, sign = True )
        if img_urlname[0] == False:
            if alt_file is None:
                print >> sys.stderr, "File", img_file, "not found"
                return None

            infile = alt_file
            outfile = img_urlname[4]

            if os.path.exists( outfile ) and os.path.getmtime( infile ) <= os.path.getmtime( outfile ):
                if self.__verbose:
                    print >> sys.stderr, "Reusing thumbnail for", infile
            else:
                img_width = 128
                if attr is not None:
                    for a in attr:
                        if a.startswith( "width=" ):
                            img_width = int( a[6:] )

                cache_name = hashlib.sha256( open( infile, 'rb' ).read() ).hexdigest()
                cache_name += "_%d" % ( img_width )

                cache_dir = os.path.join( self.__cache, "pic" )
                cache_fullname = os.path.join( self.__cache, "pic", cache_name )

                if not os.path.exists( cache_dir ):
                    os.makedirs( cache_dir )

                trigger_convert = True
                if os.path.exists( cache_fullname ):
                    if os.path.getmtime( infile ) <= os.path.getmtime( cache_fullname ):
                        if self.__verbose:
                            print >> sys.stderr, "Found thumbnail in cache, using instead"
                        trigger_convert = False

                if trigger_convert:
                    cmd = [ "convert",
                            infile,
                            "-scale", "%d" % ( img_width ),
                            cache_fullname ]

                    if self.__verbose:
                        print >> sys.stderr, "Creating thumbnail for", infile

                    subprocess.call( cmd )

                shutil.copy2( cache_fullname, outfile )

            self.sign_file( outfile )

            self.remove_from_list( outfile )

            img_urlname = [ True ] + list( img_urlname[1:] )

        if os.path.exists( img_urlname[4] ):
            cmd = [ "file", img_urlname[4] ]

            res = subprocess.Popen( cmd, stdout = subprocess.PIPE )
            ( out, err ) = res.communicate()

            m = re.match( ".*image.*, ([0-9]+) x ([0-9]+),.*", out )
            if m:
                img_width = int( m.groups()[0] )
                img_height = int( m.groups()[1] )
            else:
                cmd = [ "xli", "-identify", img_urlname[4] ]

                res = subprocess.Popen( cmd, stdout = subprocess.PIPE )
                ( out, err ) = res.communicate()

                m = re.match( ".* is a ([0-9]+)x([0-9]+).*", out )
                if m:
                    img_width = int( m.groups()[0] )
                    img_height = int( m.groups()[1] )
                    if self.__verbose:
                        print >> sys.stderr, "Got jpg size %d,%d" % ( img_width, img_height)
                else:
                    img_width = 80
                    img_height = 64
        else:
            img_width = 80
            img_height = 64

        if img_urlname[0] == True:

            img_attrs = ""

            if attr is not None:
                for a in attr:
                    if a == "align=right":
                        img_attrs = "align=\"right\""

            link_text = "<img src=\"%s\" width=\"%d\" height=\"%d\" border=\"0\" alt=\"\" %s/>" % ( img_urlname[1],
                                                                                                    img_width,
                                                                                                    img_height,
                                                                                                    img_attrs )
        else:
            link_text = None

        return link_text

    def replace_special_chars( self, line ):
        res = ""

        for char in line:
            if char in self.__char_replace:
                res += self.__char_replace[char];
            else:
                res += char

        return res

    def get_taglist( self, current_menu_entry, tag = None ):
        res = ""

        if tag:
            if tag in self.__tag_info:
                res += '<div class="taglist">'
                res += '<ul class="tag-ul">'
                for entry in self.__tag_info[tag].getEntries():
                    res += '<li>'
                    res += self.replace_link( current_menu_entry, "/" + entry.menuEntry() )
                    if entry.title():
                        res += ": " + entry.title()
                    res += '</li>'
                res += '</ul></div>'
        else:
            l1 = [ ( t, len( self.__tag_info[t].getEntries() ) ) for t in self.__tag_info.keys() ]
            l1.sort( key = lambda e: e[1], reverse = True )

            res += '<div class="taglist">'
            res += '<ul class="tag-ul">'
            for ( tag, count ) in l1:
                res += '<li>'
                res += self.replace_link( current_menu_entry, "/by-tags/" + tag + "|" + tag )
                res += ' (' + str( count ) + ')</li>'
            res += '</ul></div>'
        
        return res

    def replace_link( self, current_menu_entry, link, target_menu_entry = None ):
        a = link.split( "|" )

        if a[0].startswith( "img:" ):
            target = a[0]

            if len( a ) > 1:
                attr = a[1].split( ":" )
            else:
                attr = []
        else:
            if len( a ) < 2:
                target = a[0]
                title = a[0]
            else:
                target = a[0]
                title = a[1]

            if len( a ) > 2:
                attr = a[2].split( ":" )
            else:
                attr = []

        link_str = ""

        a_attrs = ""

        use_menu_entry = current_menu_entry
        if target_menu_entry:
            use_menu_entry = target_menu_entry

        if attr is not None:
            for a in attr:
                if a.startswith( "name=" ):
                    a_attrs = "name=\"%s\"" % ( a[5:] )

        if target.startswith( "http:" ) or target.startswith( "mailto:" ) or target.startswith( "ftp:" ) or target.startswith( "https:" ):
            if title.startswith( "img:" ):
                link_text = self.get_img_tag( current_menu_entry, title[4:], attr, target_menu_entry = target_menu_entry )
                if link_text is None:
                    link_text = title
            else:
                link_text = title

            link_str = "<a href=\""
            link_str += target
            link_str += "\" %s>" % ( a_attrs )
            link_str += link_text
            link_str += "</a>"
        elif target.startswith( "file:" ):
            filename = target[5:]
            if len( a ) < 2:
                title = filename

            full_filename = self.get_full_filename( filename, current_menu_entry )

            target_urlname = self.get_and_copy_file( use_menu_entry, full_filename, attr, sign = True )
            target_signame = self.get_and_copy_file( use_menu_entry, full_filename + ".asc", attr )

            if target_urlname[0] == True:

                if title.startswith( "img:" ):
                    img_file = title[4:]

                    link_text = self.get_img_tag( current_menu_entry, img_file, attr, target_urlname[4], target_menu_entry = target_menu_entry )

                else:
                    link_text = title
                    link_text += " (%s kB)" % ( target_urlname[2] / 1024 )

                link_str = "<a href=\""
                link_str += target_urlname[1]
                link_str += "\" %s>" % ( a_attrs )
                link_str += link_text
                link_str += "</a>"

                if target_signame[0] == True:
                    link_str += "<a href=\""
                    link_str += target_signame[1]
                    link_str += "\">"
                    link_str += " (signature)"
                    link_str += "</a>"
            else:
                print >> sys.stderr, "File not found for link", target
        elif target.startswith( "img:" ):
            filename = target[4:]

            link_text = self.get_img_tag( current_menu_entry, filename, attr, target_menu_entry = target_menu_entry )

            if link_text is not None:

                link_str = link_text
            else:
                print >> sys.stderr, "File not found for link", target
        elif target.startswith( "res:" ):
            full_target = target[4:]

            if title.startswith( "img:" ):
                link_text = self.get_img_tag( current_menu_entry, title[4:], attr, target_menu_entry = target_menu_entry )
                if link_text is None:
                    link_text = title
            else:
                link_text = title

            link_str = "<a href=\""
            link_str += full_target
            link_str += "\" %s>" % ( a_attrs )
            link_str += link_text
            link_str += "</a>"
        elif len( a ) == 1 and target[0] == '#':
            link_str = "<a name=\""
            link_str += target[1:]
            link_str += "\"></a>"
        elif target == "taglist":
            link_str = self.get_taglist( current_menu_entry )
        elif target.startswith( "taglist:" ):
            taglist = target[8:]
            link_str = self.get_taglist( current_menu_entry, taglist )
        else:
            if target.find( "#" ) >= 0:
                actual_target = target[:target.find( "#" )]
                anchor = target[target.find( "#" ) + 1:]
            else:
                actual_target = target
                anchor = ""

            if actual_target.startswith( "/" ):
                full_target = os.path.join( self.get_relative_root_dir( use_menu_entry ),
                                            actual_target[1:] )
            else:
                full_target = actual_target

            if title.startswith( "img:" ):
                link_text = self.get_img_tag( current_menu_entry, title[4:], attr, target_menu_entry = target_menu_entry )
                if link_text is None:
                    link_text = title
            else:
                link_text = title

            link_str = "<a href=\""
            link_str += full_target
            link_str += ".html"

            if len( anchor ) > 0:
                link_str += "#"
                link_str += anchor

            link_str += "\" %s>" % ( a_attrs )
            link_str += link_text
            link_str += "</a>"

        return self.replace_special_chars( link_str )

    def replace_text_formatting( self, line ):
        mode = [ "none" ]
        current_line_str = ""
        next_escaped = False

        for char in line:
            if next_escaped:
                current_line_str += char
                next_escaped = False
            elif char == '\\':
                next_escaped = True
            elif char == '*':
                if mode[-1] != "bold":
                    mode.append( "bold" )
                    current_line_str += "<b>"
                elif mode[-1] == "bold":
                    current_line_str += "</b>"
                    mode = mode[:-1]
            elif char == '_':
                if mode[-1] != "underline":
                    mode.append( "underline" )
                    current_line_str += "<u>"
                elif mode[-1] == "underline":
                    current_line_str += "</u>"
                    mode = mode[:-1]
            else:
                current_line_str += char

        return self.replace_special_chars( current_line_str )

    def replace_links( self, current_menu_entry, line, target_menu_entry = None ):
        in_link = False
        current_link_str = ""
        current_temp_line_str = ""
        current_line_str = ""
        next_escaped = False

        for char in line:
            if next_escaped:
                if char == '[' or char == ']':
                    current_temp_line_str += char
                else:
                    current_temp_line_str += '\\' + char
                next_escaped = False
            elif char == '\\':
                next_escaped = True
            elif char == '[':
                in_link = True
                current_link_str = ""
            elif char == ']' and in_link == True:
                in_link = False
                current_line_str += self.replace_text_formatting( current_temp_line_str )
                current_line_str += self.replace_link( current_menu_entry, current_link_str, target_menu_entry )

                current_temp_line_str = ""
            else:
                if in_link:
                    current_link_str += char
                else:
                    current_temp_line_str += char

        current_line_str += self.replace_text_formatting( current_temp_line_str )

        return current_line_str

    def strip_links( self, line ):
        while True:
            m = re.match( "^([^\[]*)\[[^\]]+\](.*)$", line )
            if not m:
                break

            line = m.groups()[0] + m.groups()[1]
        return line

    def get_tag_count( self, tag ):
        if tag in self.__tag_info:
            return len( self.__tag_info[tag].getEntries() )
        return 0

    def setup_tags( self, current_menu_entry, tags, target_menu_entry = None ):
        line = '<div class="file-tag">tags:'

        for tag in tags:
            count = self.get_tag_count( tag )
            tag_link = self.replace_link( current_menu_entry,
                                          "/by-tags/" + tag + "|" + tag + "<sup class=\"tag-count\">" + str( count ) + "</sup>",
                                          target_menu_entry = target_menu_entry )
            line += ' <span>' + tag_link + '</span>'

        line += '</div>'

        return line

    def generate_blog_overview( self, current_menu_entry ):
        content = "<div class=\"blog-overview\">"

        for blog_entry in self.__blog_entries[:3]:
            ( file_content, tags ) = self.read_file_content_list( [ blog_entry[0] ] )

            number_headings = False
            if "number_headings" in tags:
                if tags["number_headings"] in ( "yes", "true" ):
                    number_headings = True

            options = { "number_headings" : number_headings }

            file_content = self.transform_file_content( blog_entry[0],
                                                        file_content,
                                                        options,
                                                        target_menu_entry = current_menu_entry )

            file_content_str = ""
            for line in file_content:
                file_content_str += line + "\n"

            link = self.replace_link( current_menu_entry, "/" + blog_entry[0] + "|" + blog_entry[1] )

            content += "<div class=\"blog-title\"><h2>" + link

            if "title" in tags:
                content += ": " + tags["title"] + "</h2></div>"
            else:
                content += ":</h2></div>"
            content += "<div class=\"blog-content\">"
            content += file_content_str
            content += "</div>"

        content += "</div>"

        return content

    def generate_blog_list( self, current_menu_entry ):
        content = "<ul class=\"blog-list\">"

        for blog_entry in self.__blog_entries:
            ( file_content, tags ) = self.read_file_content_list( [ blog_entry[0] ] )

            link = self.replace_link( current_menu_entry, "/" + blog_entry[0] + "|" + blog_entry[1] )
            content += "<li>" + link
            
            if "title" in tags:
                content += ": " + tags["title"]

            content += "</li>"

        content += "</ul>"

        return content

    def generate_blog_nav( self, current_menu_entry, options, target_menu_entry = None ):
        if not "blog-date" in options:
            return ""

        found_pos = -1
        for pos, entry in enumerate( self.__blog_entries ):
            if entry[1] == options["blog-date"]:
                found_pos = pos
                break

        if found_pos < 0:
            return ""

        content = "<div class=\"blog-nav\">"

        if found_pos > 0:
            link_str = self.replace_link( current_menu_entry, "/" + self.__blog_entries[found_pos - 1][0] + "|newer (" + self.__blog_entries[found_pos - 1][1] + ")",
                                          target_menu_entry = target_menu_entry )
            link_str += ": " + self.__blog_entries[found_pos - 1][2]
            content += "<div class=\"blog-newer\">" + link_str + "</div>"

        content += " "

        if found_pos < len( self.__blog_entries ) - 1:
            link_str = self.replace_link( current_menu_entry, "/" + self.__blog_entries[found_pos + 1][0] + "|older (" + self.__blog_entries[found_pos + 1][1] + ")",
                                          target_menu_entry = target_menu_entry )
            link_str += ": " + self.__blog_entries[found_pos + 1][2]
            content += "<div class=\"blog-older\">" + link_str + "</div>"

        content += "</div>"

        return content

    def transform_file_content( self, current_menu_entry, content, options, target_menu_entry = None ):
        new_content = []

        current_list_stack = []

        header_number = []
        toc = []

        insert_toc_at_pos = -1
        toc_depth_limit = 5

        for line in content:

            h = re.match( "^=(={1,6}) (.*)$", line )
            ol = re.match( "^((?:#|\.)+) (.*)$", line )
            ul = re.match( "^((?:-|\*)+) (.*)$", line )
            ul2 = re.match( "^( +)(?:-|\*) (.*)$", line )
            ol2 = re.match( "^( +)(?:#|\.) (.*)$", line )

            rawinc = re.match( "^ *\[includeraw:(.*)\]$", line )
            insert_toc = re.match( "^\[toc(.*)\]$", line )
            tag = re.match( "^\[tag:(.*)\]$", line )
            blog_overview = re.match( "^\[blog-overview\]$", line )
            blog_list = re.match( "^\[blog-list\]$", line )
            blog_nav = re.match( "^\[blog-nav\]$", line )

            tags = []

            if h:

                hlevel = len( h.groups()[0] )

                while len( header_number ) < hlevel - 1:
                    header_number.append( 0 )

                if len( header_number ) < hlevel:
                    header_number.append( 1 )
                else:
                    header_number[hlevel - 1] += 1

                header_number = header_number[:hlevel]

                tline = ""
                aname = "h"

                initial_missing = True

                for p in range( hlevel ):
                    if header_number[p] != 0:
                        initial_missing = False

                    if initial_missing == False or header_number[p] != 0:
                        aname += "-%d" % ( header_number[p] )

                if "number_headings" in options and options["number_headings"]:

                    initial_missing = True

                    for p in range( hlevel ):

                        if header_number[p] != 0:
                            initial_missing = False

                        if initial_missing == False or header_number[p] != 0:
                            tline += "%d." % ( header_number[p] )

                    tline += " %s" % ( h.groups()[1] )
                else:
                    tline += "%s" % ( h.groups()[1] )

                new_line = "<h%d><a name=\"%s\"></a>" % ( hlevel, aname )

                new_line += tline

                toc.append( ( self.strip_links( tline ), aname, hlevel, header_number[hlevel - 1] ) )

                new_line += "</h%d>" % ( hlevel )

            elif ul or ol or ul2 or ol2:
                if ul:
                    depth = len( ul.groups()[0] )
                    new_line = ul.groups()[1]
                    list_type = "ul"
                elif ol:
                    depth = len( ol.groups()[0] )
                    new_line = ol.groups()[1]
                    list_type = "ol"
                elif ol2:
                    depth = len( ol2.groups()[0] )
                    new_line = ol2.groups()[1]
                    list_type = "ol"
                else:
                    depth = len( ul2.groups()[0] )
                    new_line = ul2.groups()[1]
                    list_type = "ul"

                if len( current_list_stack ) < 1:
                    if depth == 1:
                        # new initial list

                        current_list_stack.append( ( depth, list_type ) )

                        new_line = "<%s><li>" % ( list_type ) + new_line
                else:
                    if depth == current_list_stack[-1][0]:
                        # next entry of same depth
                        new_line = "</li><li>" + new_line
                    elif ( ( ol or ul ) and depth == current_list_stack[-1][0] + 1 ) or \
                            ( ( ol2 or ul2 ) and depth >= current_list_stack[-1][0] + 2 ):
                        # new list

                        current_list_stack.append( ( depth, list_type ) )

                        new_line = "<%s><li>" % ( list_type ) + new_line
                    elif depth < current_list_stack[-1][0]:
                        # closed list and new entry of corresponding depth

                        closing = ""
                        while depth < current_list_stack[-1][0]:
                            closing += "</li></%s>" % ( current_list_stack[-1][1] )
                            current_list_stack = current_list_stack[:-1]
                        new_line = closing + "</li><li>" + new_line
            elif len( line ) < 1 and len( current_list_stack ) > 0:
                # closed list

                closing = ""
                while len( current_list_stack ) > 0:
                    closing += "</li></%s>" % ( current_list_stack[-1][1] )
                    current_list_stack = current_list_stack[:-1]
                new_line = closing

                current_list_stack = []
            elif rawinc:
                rawfile = rawinc.groups()[0]
                if os.path.exists( rawfile ):
                    fh = open( rawfile )
                    for rawline in fh:
                        new_content.append( rawline )
                    continue
            elif insert_toc:
                insert_toc_at_pos = len( new_content )
                all_attr = insert_toc.groups()[0]
                if all_attr and len( all_attr ) > 0:
                    for attr in all_attr.split( '|' ):
                        if attr.startswith( "limit=" ):
                            toc_depth_limit = int( attr.split( '=' )[1] )
            elif tag:
                file_tags = tag.groups()[0]
                if file_tags and len( file_tags ) > 0:
                    for file_tag in file_tags.split( ',' ):
                        tags.append( file_tag )
            else:
                new_line = line

            if blog_overview:
                new_line = self.generate_blog_overview( current_menu_entry )
            elif blog_list:
                new_line = self.generate_blog_list( current_menu_entry )
            elif blog_nav:
                new_line = self.generate_blog_nav( current_menu_entry, options, target_menu_entry )
            elif len( tags ) > 0:
                new_line = self.setup_tags( current_menu_entry, tags, target_menu_entry = target_menu_entry )
            else:
                new_line = self.replace_links( current_menu_entry, new_line, target_menu_entry = target_menu_entry )

            new_content.append( new_line )

        if len( current_list_stack ) > 0:
            # close list

            closing = ""
            while len( current_list_stack ) > 0:
                closing += "</li></%s>" % ( current_list_stack[-1][1] )
                current_list_stack = current_list_stack[:-1]
            new_line = closing

            current_list_stack = []

            new_content.append( new_line )

        if insert_toc_at_pos >= 0:
            new_content_with_toc = new_content[0:insert_toc_at_pos]

            current_level = toc[0][2]
            start_level = current_level

            new_line = "<div class=\"toc\"><ul>"
            new_content_with_toc.append( new_line )
            for toc_entry, toc_aname, toc_level, toc_number in toc:

                if toc_level == current_level:

                    if toc_number == 1:
                        new_line = "<li class=\"toc_entry\">"
                    else:
                        new_line = "</li><li class=\"toc_entry\">"

                    new_line += "<a href=\"#%s\">%s</a>" % ( toc_aname, toc_entry )
                elif toc_level > current_level:
                    if toc_level <= toc_depth_limit:
                        new_line = ""
                        while current_level < toc_level:
                            new_line += "<ul><li class=\"toc_entry\">"
                            current_level += 1

                        new_line += "<a href=\"#%s\">%s</a>" % ( toc_aname, toc_entry )

                        current_level = toc_level
                    else:
                        continue
                else:
                    new_line = ""
                    while current_level > toc_level:
                        new_line += "</li></ul>"
                        current_level -= 1
                    new_line += "</li><li class=\"toc_entry\"><a href=\"#%s\">%s</a>" % ( toc_aname, toc_entry )

                    current_level = toc_level

                new_content_with_toc.append( new_line )

            new_line = ""
            while current_level > ( start_level - 1 ):
                new_line += "</li></ul>"
                current_level -= 1

            new_line += "</div>"
            new_content_with_toc.append( new_line )

            new_content_with_toc += new_content[insert_toc_at_pos + 1:]

            new_content = new_content_with_toc

        return new_content

    def extract_tags( self, current_menu_entry, content ):
        tags = []

        for line in content:
            tag = re.match( "^\[tag:(.*)\]$", line )

            if tag:
                file_tags = tag.groups()[0]
                if file_tags and len( file_tags ) > 0:
                    for file_tag in file_tags.split( ',' ):
                        tags.append( file_tag )

        return tags

    def add_tag_info( self, current_menu_entry, tag, title ):
        ti = self.__tag_info.get( tag, TagInfo( tag ) )
        ti.addEntry( current_menu_entry, title )
        self.__tag_info[tag] = ti

    def sign_file( self, filename ):
        if self.__sign_files:
            create_signature = False

            if not os.path.exists( filename + ".asc" ):
                create_signature = True
            else:
                cmd = [ "gpg",
                        "--batch",
                        "--verify",
                        filename + ".asc" ]
            
                try:
                    subprocess.check_output( cmd, stderr = subprocess.STDOUT )
                except subprocess.CalledProcessError as e:
                    create_signature = True

            if create_signature:
                cache_name = hashlib.sha256( open( filename, 'rb' ).read() ).hexdigest()
                cache_name += ".asc"

                cache_dir = os.path.join( self.__cache, "asc" )
                cache_fullname = os.path.join( self.__cache, "asc", cache_name )

                if not os.path.exists( cache_dir ):
                    os.makedirs( cache_dir )

                if not os.path.exists( cache_fullname ):
                    cmd = [ "gpg",
                            "-ab",
                            "--batch",
                            "--no-tty",
                            "-o",
                            cache_fullname ]

                    if self.__sign_key:
                        cmd.extend( [ "--default-key",
                                      self.__sign_key ] )

                    cmd.append( filename )

                    subprocess.check_output( cmd, stderr = subprocess.STDOUT )

                shutil.copy2( cache_fullname, filename + ".asc" )
    
    def generate_html( self, basedir, filename, menus, groupmenus, template = None, template_replacements = None, stage = None ):
        base_without_cms = basedir[ len( self.__cms_dir ):]
        if len( base_without_cms ) > 0 and base_without_cms[0] == "/":
            base_without_cms = base_without_cms[1:]

        target_dir = os.path.join( self.__output_dir, base_without_cms )
        target_file = os.path.join( target_dir, filename[:-4] + ".html" )

        try:
            os.makedirs( target_dir )
        except:
            pass

        current_menu_entry = os.path.join( basedir, filename[:-4] )[len(self.__cms_dir)+1:]

        if current_menu_entry.endswith( "/" ):
            base_url = "../" * ( current_menu_entry.count( "/" ) - 1 )
        else:
            base_url = "../" * current_menu_entry.count( "/" )

        ( parent_files, menu_content, menu_toplevel, menu_submenu ) = self.generate_menu( "", menus, current_menu_entry, current_menu_entry, groupmenus )

        use_template = False
        if template and os.path.exists( os.path.join( self.__cms_dir, template ) + ".txt" ):
            use_template = True
            parent_files = [ template ]

        ( file_content, tags ) = self.read_file_content_list( parent_files )

        if template_replacements:
            file_content = [ l.format( **template_replacements ) for l in file_content ]

        number_headings = False
        if "number_headings" in tags:
            if tags["number_headings"] in ( "yes", "true" ):
                number_headings = True

        options = { "number_headings" : number_headings }

        file_tags = self.extract_tags( current_menu_entry, file_content )

        for tag in file_tags:
            self.add_tag_info( current_menu_entry, tag, tags.get( "title", None ) )

        if stage and stage == 1:
            if "blog-date" in tags:
                self.__blog_entries.append( ( current_menu_entry, tags["blog-date"], tags.get( "title", None ) ) )
            return

        if "blog-date" in tags:
            options["blog-date"] = tags["blog-date"]

        file_content = self.transform_file_content( current_menu_entry, file_content, options )

        fh = open( target_file, "w" )

        self.remove_from_list( target_file )

        if "title" in tags:
            title = tags["title"]
        else:
            title = "N/A"

        file_content_str = ""
        for line in file_content:
            file_content_str += line + "\n"

        if groupmenus:
            group_menu_content = self.generate_groupmenu( current_menu_entry, groupmenus, current_menu_entry )
        else:
            group_menu_content = ""

        sig_link_str = ""

        if self.__sign_files:
            sig_link_str = "<a class=\"webpage_sig\" href=\"%s\">sig</a>" % ( filename[:-4] + ".html.asc" )

        replace_keywords = { "title" : title,
                             "css_name" : base_url + self.__base_css,
                             "menu_content" : menu_content,
                             "menu_toplevel" : menu_toplevel,
                             "menu_submenu" : menu_submenu,
                             "groupmenu_content" : group_menu_content,
                             "file_content" : file_content_str,
                             "sig_link" : sig_link_str }

        tpl_fh = open( self.__tpl_file, "r" )
        tpl_content = tpl_fh.read()

        print >> fh, tpl_content.format( **replace_keywords )

        fh.close()

        self.sign_file( target_file )

        if self.__validate_html:
            cmd = [ "tidy",
                    "-e",
                    "-q",
                    target_file ]

            try:
                subprocess.check_output( cmd, stderr = subprocess.STDOUT )
            except subprocess.CalledProcessError as e:
                print >> sys.stderr, "HTML problems for %s:" % ( target_file )
                print >> sys.stderr, e.output

    def remove_from_list( self, path ):
        to_remove = []

        for p in self.__old_output_entries:
            if p.startswith( path ):
                to_remove.append( p )

        for p in to_remove:
            self.__old_output_entries.remove( p )

def print_version( option, opt_str, value, parser ):
    print "mmu2html.py %s" % ( MMU2HTML_VERSION )
    sys.exit( 0 )

if __name__ == "__main__":
    parser = OptionParser( usage = "usage: %prog [options] [<files>]" )
    parser.add_option( "-c", "--config",
                       action  = "store",
                       type    = "string",
                       dest    = "config",
                       help    = "config" )
    parser.add_option( "-d", "--delete",
                   action  = "store_true",
                   dest    = "delete_stale",
                   default = False,
                   help    = "delete stale output files" )
    parser.add_option( "--validate",
                   action  = "store_true",
                   dest    = "validate_html",
                   default = False,
                   help    = "validate generate HTML files with tidy" )
    parser.add_option( "-v", "--verbose",
                   action  = "store_true",
                   dest    = "verbose",
                   default = False,
                   help    = "verbose output" )
    parser.add_option( "-V", "--version",
                       action  = "callback",
                       callback = print_version,
                       help    = "print mmu2html version" )


    ( options, args ) = parser.parse_args()

    config = ConfigParser.ConfigParser()
    config.read( options.config )

    m = MMU2Html( config )

    m.set_delete_stale( options.delete_stale )
    m.set_validate_html( options.validate_html )
    m.set_verbose_output( options.verbose )

    m.generate()
