#! /usr/bin/env python
# past.py
# Copyright (C) 2013-2014 Ralf Hoffmann.
# ralf@boomerangsworld.de
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import datetime
import time
import math

PAST_VERSION="0.1.1"

class Resource:
    def __init__( self, name, avail = 100 ):
        self.__name = name
        self.__avail = avail

    def getAvail( self ):
        return self.__avail

    def getName( self ):
        return self.__name

class Task:
    def __init__( self, name, res, effort, completed, depends = None, prio = 0 ):
        self.__name = name
        self.__res = res
        self.__effort = effort
        self.__completed = completed
        self.__depends = depends
        self.__prio = prio
        self.__finish_date = None
        self.__actual_prio = 0

    def getName( self ):
        return self.__name

    def getDepends( self ):
        return self.__depends

    def getEffort( self ):
        return self.__effort

    def getResource( self ):
        return self.__res

    def getCompleted( self ):
        return self.__completed

    def setFinishDate( self, date ):
        self.__finish_date = date

    def getFinishDate( self ):
        return self.__finish_date

    def getPriority( self ):
        return self.__prio

    def setActualPrio( self, v ):
        self.__actual_prio = v

    def getActualPrio( self ):
        return self.__actual_prio

class Milestone:
    def __init__( self, name, date, buf ):
        self.__name = name

        if isinstance( date, str):
            self.__date = datetime.date.fromtimestamp( time.mktime( time.strptime( date,
                                                                                   "%Y-%m-%d" ) ) )
        else:
            self.__date = date

        self.__buffer = buf
        self.__tasks = []
        self.__milcon = 0
        self.__completed = 0

    def getTasks( self ):
        for t in self.__tasks:
            yield t

    def add( self, task ):
        self.__tasks.append( task )

    def getDate( self ):
        return self.__date

    def getBuffer( self ):
        return self.__buffer

    def getName( self ):
        return self.__name

    def setMILCON( self, level ):
        self.__milcon = level

    def getMILCON( self ):
        return self.__milcon

    def setTotalCompleted( self, val ):
        self.__completed = val

    def getTotalCompleted( self ):
        return self.__completed

    def assignActualPrio( self, task, prio ):
        if task.getActualPrio() < 0 or task.getActualPrio() < prio:
            task.setActualPrio( prio )

        if task.getDepends():
            self.assignActualPrio( task.getDepends(), prio + 1 )

    def calculateActualPriorities( self ):
        min_prio = None
        max_prio = None

        for t in self.__tasks:
            p = t.getPriority()
            t.setActualPrio( -1 )

            if min_prio is None or p < min_prio:
                min_prio = p
            if max_prio is None or p > max_prio:
                max_prio = p

        prio = min_prio

        while prio <= max_prio:
            for t in self.__tasks:
                if t.getPriority() == prio:
                    self.assignActualPrio( t, prio - min_prio )

            prio += 1

class PAST:
    def __init__( self, starttime ):
        self.__schedule = {}
        self.__scheduled_tasks = {}

        if isinstance( starttime, str):
            self.__starttime = datetime.date.fromtimestamp( time.mktime( time.strptime( starttime,
                                                                                   "%Y-%m-%d" ) ) )
        else:
            self.__starttime = starttime

        self.__biggest_time = self.__starttime
        self.__vacation_task = Task( "NW", None, 1, 0 )
        self.__milestones = []

    def __reserveDate( self, res, date ):
        if res not in self.__schedule:
            self.__schedule[res] = {}
        
        if date.weekday() < 5:
            self.__schedule[res][date] = self.__vacation_task

    def __isVacationTask( self, task ):
        return task == self.__vacation_task
                
    def __placeTask( self, task, duration, res, not_earlier = None ):
        if res not in self.__schedule:
            self.__schedule[res] = {}
        
        t = self.__starttime

        if not_earlier is not None:
            t = not_earlier

        while t.weekday() >= 5:
            t += datetime.timedelta( 1 )
        
        if duration > 0:
            while duration > 0:
                while t in self.__schedule[res]:
                    t += datetime.timedelta( 1 )

                    while t.weekday() >= 5:
                        t += datetime.timedelta( 1 )

                self.__schedule[res][t] = task
                
                duration -= 1

            if ( t > self.__biggest_time ):
                self.__biggest_time = t
            task.setFinishDate( t )
        else:
            task.setFinishDate( t )            


        # for day in range( duration ):
        #     if len( self.__schedule[res] ) == 0:
        #         self.__schedule[res] = [ ( self.__starttime, task ) ]
        #         #print "Placed at", self.__schedule[res][-1]
        #     else:
        #         last = self.__schedule[res][-1]
        #         t = last[0]
        #         t += datetime.timedelta( 1 )

        #         while t.weekday() >= 5:
        #             t += datetime.timedelta( 1 )

        #         self.__schedule[res].append( ( t, task ) )

        #         if ( t > self.__biggest_time ):
        #             self.__biggest_time = t

        #         #print "Placed at", self.__schedule[res][-
        # if duration > 0:
        #     task.setFinishDate( self.__schedule[res][-1][0] )

    def getBiggestTime( self ):
        return self.__biggest_time

    def getResources( self ):
        for r in sorted( self.__schedule.keys() ):
            yield r

    def __getPlacedTasks( self, res ):
        if res not in self.__schedule:
            yield
        for p in sorted( self.__schedule[res].keys() ):
            yield ( p, self.__schedule[res][p] )

    def getStartTime( self ):
        return self.__starttime

    def scheduleVacation( self, res, startdate, duration ):
        if isinstance( startdate, str):
            startdate = datetime.date.fromtimestamp( time.mktime( time.strptime( startdate,
                                                                                 "%Y-%m-%d" ) ) )

        while duration > 0:
            while startdate.weekday() >= 5:
                startdate += datetime.timedelta( 1 )
            if startdate >= self.getStartTime():
                self.__reserveDate( res, startdate )
            startdate += datetime.timedelta( 1 )
            duration -= 1

    def __scheduleTask( self, task ):
        if task in self.__scheduled_tasks:
            return

        start = None

        if task.getDepends():
            self.__scheduleTask( task.getDepends() )

            start = task.getDepends().getFinishDate()
            start += datetime.timedelta( 1 )

        duration = task.getEffort()
        duration = duration - duration * task.getCompleted() / 100

        duration *= 100
        duration /= task.getResource().getAvail()
        duration = int( math.ceil( duration ) )

        self.__placeTask( task, duration, task.getResource(), start )

        self.__scheduled_tasks[task] = 1

    def scheduleMilestone( self, milestone ):
        total_progress = 0
        total_progress_count = 0

        milestone.calculateActualPriorities()

        min_prio = None
        max_prio = None

        for t in milestone.getTasks():
            p = t.getActualPrio()

            if min_prio is None or p < min_prio:
                min_prio = p
            if max_prio is None or p > max_prio:
                max_prio = p

        prio = max_prio

        while prio >= min_prio:
            for t in milestone.getTasks():
                if t.getActualPrio() == prio:
                    self.__scheduleTask( t )

                    if t.getPriority() >= 0:
                        total_progress += t.getEffort() * t.getCompleted() / 100.0
                        total_progress_count += t.getEffort()

            prio -= 1

        if total_progress_count > 0:
            milestone.setTotalCompleted( total_progress * 100 / total_progress_count )
        else:
            milestone.setTotalCompleted( 100 )

        self.__milestones.append( milestone )

    def __checkTargetDate( self, milestone, handle ):
        t = None
        for task in milestone.getTasks():
            if task.getFinishDate() and task.getPriority() >= 0:
                if not t:
                    t = task.getFinishDate()
                elif task.getFinishDate() > t:
                    t = task.getFinishDate()

        print_text = True
        if milestone.getTotalCompleted() >= 100 and milestone.getDate() < self.getStartTime():
            print_text = False

        if print_text:
            print >> handle, "<tr><th colspan=\"1\">", milestone.getName(), "</th></tr>"

        if not t:
            t = milestone.getDate()

        if milestone.getDate() > t:
            startt = t
            endt = milestone.getDate()
            neg_days = 0
        else:
            startt = milestone.getDate()
            endt = t
            neg_days = 1

        diff_days = 0

        while startt < endt:
            while startt.weekday() >= 5:
                startt += datetime.timedelta( 1 )

            startt += datetime.timedelta( 1 )

            diff_days += 1

        if neg_days == 1:
            diff_days *= -1

        if print_text:
            print >> handle, "<tr><td></td><td>Remaining days:</td><td>", diff_days, "</td></tr>"

        buf = datetime.timedelta( milestone.getBuffer() )

        #buf_days = int( math.ceil( buf.total_seconds() / 60 / 60 / 24 ) )
        buf_days = milestone.getBuffer()

        if print_text:
            print >> handle, "<tr><td></td><td>Buffer:</td><td>", buf_days, "</td></tr>"

        ratio = ( float( diff_days ) / buf_days )

        if print_text:
            print >> handle, "<tr><td></td><td>Buffer usage:</td><td>", ratio, "</td></tr>"

        if print_text:
            print >> handle, "<tr><td></td><td>Status:</td><td>"

        if diff_days >= buf_days:
            if print_text:
                print >> handle, "<span style=\"background-color:blue;color:white\">"
                print >> handle, "MILCON 5"
                print >> handle, "</span>"
            milestone.setMILCON( 5 )
        elif ratio >= 0.8:
            if print_text:
                print >> handle, "<span style=\"background-color:green;color:white\">"
                print >> handle, "MILCON 4"
                print >> handle, "</span>"
            milestone.setMILCON( 4 )
        elif ratio >= 0.5:
            if print_text:
                print >> handle, "<span style=\"background-color:yellow;color:black\">"
                print >> handle, "MILCON 3"
                print >> handle, "</span>"
            milestone.setMILCON( 3 )
        elif ratio >= 0.2:
            if print_text:
                print >> handle, "<span style=\"background-color:red;color:white\">"
                print >> handle, "MILCON 2"
                print >> handle, "</span>"
            milestone.setMILCON( 2 )
        else:
            if print_text:
                print >> handle, "<span style=\"background-color:white;color:black\">"
                print >> handle, "MILCON 1"
                print >> handle, "</span>"
            milestone.setMILCON( 1 )

        if print_text:
            print >> handle, "</td></tr>"

    def __checkMetMilestone( self, task, milestones ):
        hit = None

        for m in milestones:
            if task in m.getTasks():
                hit = m

        if hit:
            if task.getFinishDate() > hit.getDate():
                return ( hit, 2 )

            for m in milestones:
                if task.getFinishDate() < m.getDate() and m != hit and m.getDate() < hit.getDate():
                    # finished early
                    return ( hit, 0 )
            return ( hit, 1 )
        else:
            # no milestone so alway met
            return ( None, 0 )

    def printSchedule( self, name = "plan.html" ):
        fh = open( name, "w" )

        print >> fh, """
        <html>
        <head>
        </head>
        <body>
        """

        print >> fh, """
        <table border=\"1\">
        """

        for m in self.__milestones:
            self.__checkTargetDate( m, fh )

        print >> fh, """
        </table>
        """

        smallest = None
        highest = None
        for res in self.getResources():
            for p in self.__getPlacedTasks( res ):
                if not smallest:
                    smallest = p[0]
                elif p[0] < smallest:
                    smallest = p[0]
                if not highest:
                    highest = p[0]
                elif p[0] > highest:
                    highest = p[0]
                #print p[0], p[1].getName()

        for m in self.__milestones:
            if m.getDate() > highest:
                highest = m.getDate()

        table = {}
        for res in self.getResources():
            table[res] = {}

            current_time = smallest
            while current_time <= highest:

                table[res][current_time] = None

                current_time += datetime.timedelta( 1 )

                while current_time.weekday() >= 5:
                    current_time += datetime.timedelta( 1 )

        print >> fh, "<table border=\"1\"><tr><th>Calender week</th>"

        current_time = smallest
        while current_time <= highest:
            while current_time.weekday() >= 5:
                current_time += datetime.timedelta( 1 )

            print >> fh, "<th>", current_time.isocalendar()[1], "</th>"
            #print >> fh, "<th>", current_time, "</th>"

            current_time += datetime.timedelta( 1 )

        print >> fh, "</tr>"
        print >> fh, "<tr><th>Milestone</th>"

        current_time = smallest
        while current_time <= highest:
            while current_time.weekday() >= 5:
                current_time += datetime.timedelta( 1 )

            hit = None

            for m in self.__milestones:
                if current_time == m.getDate():
                    hit = m

            if hit:
                print >> fh, "<th>", hit.getName()

                if hit.getMILCON() == 5:
                    print >> fh, "<br /><span style=\"background-color:blue;color:white\">MILCON5</span>"
                elif hit.getMILCON() == 4:
                    print >> fh, "<br /><span style=\"background-color:green;color:white\">MILCON4</span>"
                elif hit.getMILCON() == 3:
                    print >> fh, "<br /><span style=\"background-color:yellow;color:black\">MILCON3</span>"
                elif hit.getMILCON() == 2:
                    print >> fh, "<br /><span style=\"background-color:red;color:white\">MILCON2</span>"
                elif hit.getMILCON() == 1:
                    print >> fh, "<br /><span style=\"background-color:white;color:black\">MILCON1</span>"

                print >> fh, "<br />%d%%" % ( hit.getTotalCompleted() )

                print >> fh, "</th>"
            else:
                print >> fh, "<th></th>"

            current_time += datetime.timedelta( 1 )

        print >> fh, "</tr>"

        for res in self.getResources():
            print >> fh, "<tr><td>", res.getName(), "</td>"

            #current_time = smallest
            last_task = None
            last_task_len = 0

            for p in self.__getPlacedTasks( res ):
                table[res][p[0]] = p[1]

            for current_time in sorted( table[res].keys() ):
                if current_time.weekday() >= 5:
                    continue

                task = table[res][current_time]

                if task == last_task:
                    last_task_len += 1
                else:
                    if last_task_len > 0:
                        print >> fh, "<td colspan=\"", last_task_len, "\">"
                        if last_task:
                            if self.__isVacationTask( last_task ):
                                print >> fh, "vacation"
                            else:
                                print >> fh, last_task.getName(), "<br />", last_task.getCompleted(), "%"
                                if last_task.getPriority() >= 0:
                                    ( task_milestone, met ) = self.__checkMetMilestone( last_task, self.__milestones )
                                    if met == 2:
                                        print >> fh, "<br />LATE: ", task_milestone.getName()
                                    elif met == 0:
                                        print >> fh, "<br />EARLY: ", task_milestone.getName()
                                else:
                                    print >> fh, "<br />LOW"
                        print >> fh, "</td>"

                    last_task = task
                    last_task_len = 1

            if last_task_len > 0:
                print >> fh, "<td colspan=\"", last_task_len, "\">"
                if last_task:
                    if self.__isVacationTask( last_task ):
                        print >> fh, "vacation"
                    else:
                        print >> fh, last_task.getName(), "<br />", last_task.getCompleted(), "%"

                        if last_task.getPriority() >= 0:
                            ( task_milestone, met ) = self.__checkMetMilestone( last_task, self.__milestones )
                            if met == 2:
                                print >> fh, "<br />LATE: ", task_milestone.getName()
                            elif met == 0:
                                print >> fh, "<br />EARLY: ", task_milestone.getName()
                        else:
                            print >> fh, "<br />LOW"
                print >> fh, "</td>"

            print >> fh, "</tr>"

        print >> fh, """
        </table>
        </body>
        </html>
        """

if __name__ == "__main__":

    d1 = Resource( "dev1", 30 )
    d2 = Resource( "dev2", 100 )

    m1 = Milestone( "v1", date( 2013, 9, 9 ), 5 )
    m1.add( Task( "t1", d1, 5, 0 ) )
    m1.add( Task( "t2", d2, 15, 25 ) )
    m1.add( Task( "t3", d2, 5, 0 ) )
    m1.add( Task( "t4", d2, 5, 0 ) )

    s = PAST( date( 2013, 7, 29 ) )
    s.scheduleVacation( d2, date( 2013, 7, 30 ), 5 )

    s.scheduleMilestone( m1 )

    s.printSchedule()
