/* main.cc
 * This file belongs to Runner.
 * Copyright (C) 2012 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/runner
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "runner.hh"
#include "worker/nwc_path.hh"

#ifdef HAVE_GETOPT_LONG
#  ifndef _GNU_SOURCE
#    define _GNU_SOURCE
#    include <getopt.h>
#    undef _GNU_SOURCE
#  else
#    include <getopt.h>
#  endif
#endif

#include <iostream>

#define RUNNER_VERSION "0.1"

int main( int argc, char **argv )
{
    bool showversion, showhelp;
    std::string configbasedir;

    showversion = false;
    showhelp = false;

    while ( true ) {
#ifdef HAVE_GETOPT_LONG
        int option_index;
        static struct option long_options[] = {
            { "help", no_argument, NULL, 'h' },
            { "version", no_argument, NULL, 'V' },
            { "config", required_argument, NULL, 'c' },
            { NULL, 0, NULL, 0 }
        };
#endif

#ifdef HAVE_GETOPT_LONG
        int c = getopt_long( argc, argv, "hVc:", long_options, &option_index );
#else
        int c = getopt( argc, argv, "hVc:" );
#endif

        if ( c == -1 ) break;
      
        switch( c ) {
            case 'c':
                if ( optarg != NULL ) {
                    configbasedir = optarg;
                    if ( NWC::Path::isAbs( configbasedir ) ) {
                    } else {
                        fprintf( stderr, "Runner: relative path %s not allowed for config base directory\n", configbasedir.c_str() );
                        configbasedir.clear();
                    }
                }
                break;
            case 'h':
                showhelp = true;
                break;
            case 'V':
                showversion = true;
                break;
            default:
                break;
        }
    }

    if ( showversion == true ) {
        std::cout << "Runner version: " << RUNNER_VERSION << std::endl;
        std::cout << "  Author: Ralf Hoffmann <ralf@boomerangsworld.de>" << std::endl;
        std::cout << "  Homepage: http://www.boomerangsworld.de/runner" << std::endl;
        exit( EXIT_SUCCESS );
    }

    if ( showhelp == true ) {
        std::cout << "Runner Menu" << std::endl;
        std::cout << "  Version: " << RUNNER_VERSION << std::endl;
        std::cout << "  Author: Ralf Hoffmann <ralf@boomerangsworld.de>" << std::endl;
        std::cout << "  Homepage: http://www.boomerangsworld.de/runner" << std::endl;
        std::cout << "  Options: -V, --version\tPrint the program version and exit" << std::endl;
        std::cout << "           -h, --help\t\tPrint this help and exit" << std::endl;
        std::cout << "           -c, --config=<path>\tUse given path for configuration data" << std::endl;
        exit( EXIT_SUCCESS);
    }
 
    RunnerUI ui( argc, argv );

    if ( ! configbasedir.empty() ) {
        ui.setConfigDir( configbasedir );
    }

    ui.mainLoop();

    return 0;
}
