/* runner.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2012-2020 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "runner.hh"
#include <aguix/popupmenu.hh>
#include <aguix/button.h>
#include <aguix/fieldlistview.h>
#include <aguix/stringgadget.h>
#include <aguix/cyclebutton.h>
#include "worker/pers_string_list.hh"
#include "worker/nwc_path.hh"
#include "worker/nwc_dir.hh"
#include "worker/nwc_file.hh"

#include <list>
#include <iostream>
#include <fstream>
#include <algorithm>

static std::list< std::pair< std::string, std::string > > parse_config_file( const std::string &config_file )
{
    std::ifstream ifs( config_file );

    std::list< std::pair< std::string, std::string > > v;

    if ( ! ifs.is_open() ) return v;

    while ( ! ifs.eof() ) {
        std::string s;

        std::getline( ifs, s );

        if ( ! s.empty() &&
             s[0] != '[' && s[0] != '#' ) {

            const std::string::size_type pos = s.find( '=' );

            if ( pos != std::string::npos && pos > 0 && pos + 1 < s.length() ) {
                std::string::size_type start = 0, end = pos - 1;

                while ( start < end ) {
                    if ( std::isspace( s[start] ) ) {
                        start++;
                    } else {
                        break;
                    }
                }
                while ( start < end ) {
                    if ( std::isspace( s[end] ) ) {
                        end--;
                    } else {
                        break;
                    }
                }

                std::string key( s, start, end - start + 1 );

                start = pos + 1, end = s.length() - 1;

                while ( start < end ) {
                    if ( std::isspace( s[start] ) ) {
                        start++;
                    } else {
                        break;
                    }
                }
                while ( start < end ) {
                    if ( std::isspace( s[end] ) ) {
                        end--;
                    } else {
                        break;
                    }
                }

                std::string value( s, start, end - start + 1 );

                v.push_back( std::make_pair( key, value ) );
            }
        }
    }

    return v;
}

int KnownPrograms::read( const std::string &config_file )
{
    m_known_programs.clear();

    std::list< std::pair< std::string, std::string > > v = parse_config_file( config_file );

    std::vector< std::string > names;
    std::vector< std::string > programs;

    for ( auto it0 = v.begin();
          it0 != v.end();
          it0++ ) {
        if ( it0->first == "name" ) {
            names.push_back( it0->second );
        } else if ( it0->first == "program" ) {
            programs.push_back( it0->second );
        }
    }

    for ( auto it1 = names.begin(), it2 = programs.begin();
          it1 != names.end() && it2 != programs.end();
          it1++, it2++ ) {
        m_known_programs.push_back( std::make_pair( *it1, *it2 ) );
    }

    // m_known_programs.push_back( std::make_pair( "Terminal", "xterm" ) );
    // m_known_programs.push_back( std::make_pair( "Worker", "worker" ) );
    // m_known_programs.push_back( std::make_pair( "Firefox", "firefox" ) );
    // m_known_programs.push_back( std::make_pair( "Display settings", "xfce4-display-settings" ) );
    // m_known_programs.push_back( std::make_pair( "Logout", "xfce4-session-logout" ) );

    return 0;
}

RunnerUI::RunnerUI( int argc, char **argv ) : m_argc( argc ),
                                              m_argv( argv ),
                                              m_completion_mode( HISTORY_MODE ),
                                              m_show_run_dialog( false )
{
    m_aguix = new AGUIX( argc, argv, "Runner" );
    m_aguix->initX();

    if ( m_aguix->checkX() ) {
        m_aguix->addDefaultColors();
#ifdef HAVE_XFT
        m_aguix->setFont( "sans-10" );
#endif
    } else {
        delete m_aguix;
        m_aguix = NULL;
    }
}

void RunnerUI::setConfigDir( const std::string &d )
{
    m_config_dir = d;
}

std::string RunnerUI::getConfigDir()
{
    if ( m_config_dir.empty() ) {
        const char *home = getenv( "HOME" );

        if ( home ) {
            std::string cfg = home;
            cfg += "/.runner";
            m_config_dir = cfg;
        }
    }

    if ( m_config_dir.empty() ) {
        std::cerr << "Runner: No configuration directory set" << std::endl;
        exit( EXIT_FAILURE );
    }

    NWC::FSEntry e( m_config_dir );

    if ( e.isDir( true ) == false ) {
        if ( worker_mkdir( m_config_dir.c_str(), 0700 ) != 0 ) {
            std::cerr << "Runner: Couldn't create configuration directory " << m_config_dir << std::endl;
            exit( EXIT_FAILURE );
        }
    }
    return m_config_dir;
}

RunnerUI::~RunnerUI()
{
    if ( m_aguix != NULL ) {
        delete m_aguix;
    }
}

void RunnerUI::runCommand( const std::string &cmd )
{
    if ( ! cmd.empty() ) {
        std::string c = cmd;
        c += " &";

        //TODO better background execution
        system( c.c_str() );
    }
}

void RunnerUI::updateCompletionList( const std::string &prefix )
{
    std::list< std::string > l;

    if ( m_completion_mode == PATH_MODE ) {
        findCommands();

        l = m_commands_in_path;
    } else {
        l = m_history->getList();
    }

    m_lv->setSize( 0 );

    for ( auto it1 = l.begin();
          it1 != l.end();
          it1++ ) {
        if ( AGUIXUtils::starts_with( *it1, prefix ) ) {
            int row = m_lv->addRow();

            m_lv->setText( row, 0, *it1 );
            m_lv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
        }
    }

    m_lv->redraw();
}

void RunnerUI::runProgramDialog()
{
    if ( ! m_aguix ) return;

    std::string history_file = getConfigDir();

    history_file += "/history";

    m_history = new PersistentStringList( history_file );

    m_win = new AWindow( m_aguix,
                         0, 0,
                         400, 400,
                         "Run program",
                         AWindow::AWINDOW_NORMAL );
    m_win->create();

    m_co1 = m_win->setContainer( new AContainer( m_win, 1, 4 ), true );
    m_co1->setMaxSpace( 5 );
    
    AContainer *co1_2 = m_co1->add( new AContainer( m_win, 2, 1 ), 0, 0 );
    co1_2->setMaxSpace( 5 );
    co1_2->setBorderWidth( 0 );
    co1_2->add( new Text( m_aguix, 0, 0, "Enter program:" ),
                0, 0, AContainer::CO_FIX );
    m_sg = static_cast<StringGadget*>( co1_2->add( new StringGadget( m_aguix, 0, 0, 100, "", 1 ),
                                                   1, 0, AContainer::CO_INCW ) );
    
    m_mode_cyb = static_cast< CycleButton * >( m_co1->add( new CycleButton( m_aguix, 0, 0, 100, 0 ),
                                                           0, 1, AContainer::CO_INCW ) );

    m_mode_cyb->addOption( "History (TAB: switch to all programs)" );
    m_mode_cyb->addOption( "Path completion (TAB: switch to history)" );

    m_lv = static_cast<FieldListView*>( m_co1->add( new FieldListView( m_aguix, 
                                                                       0, 0, 
                                                                       400, 300, 0 ),
                                                    0, 2, AContainer::CO_MIN ) );
    m_lv->setNrOfFields( 1 );
    //m_lv->setShowHeader( true );
    //m_lv->setFieldText( 0, "History" );
    m_lv->setHBarState( 2 );
    m_lv->setVBarState( 2 );
    //m_lv->setAcceptFocus( true );
    //m_lv->setDisplayFocus( true );

    AContainer *co1_3 = m_co1->add( new AContainer( m_win, 2, 1 ), 0, 3 );
    co1_3->setMinSpace( 5 );
    co1_3->setMaxSpace( -1 );
    co1_3->setBorderWidth( 0 );
    m_okb = (Button*)co1_3->add( new Button( m_aguix,
                                             0,
                                             0,
                                             "Run",
                                             1,
                                             0,
                                             0 ), 0, 0, AContainer::CO_FIX );
    m_cancelb = (Button*)co1_3->add( new Button( m_aguix,
                                                 0,
                                                 0,
                                                 "Cancel",
                                                 1,
                                                 0,
                                                 0 ), 1, 0, AContainer::CO_FIX );

    m_okb->setAcceptFocus( false );
    m_cancelb->setAcceptFocus( false );
    m_mode_cyb->setAcceptFocus( false );

    m_win->contMaximize( true );
    m_win->setDoTabCycling( true );

    maximizeWin();

    updateCompletionList( "" );

    m_win->show();

    m_sg->takeFocus();

    AGMessage *msg;
    int endmode = 0;

    int last_cursor_pos = 0;

    for ( ; endmode == 0; ) {
        msg = m_aguix->WaitMessage( NULL );
        if ( msg != NULL ) {
            switch ( msg->type ) {
                case AG_CLOSEWINDOW:
                    endmode = -1;
                    break;
                case AG_BUTTONCLICKED:
                    if ( msg->button.button == m_okb ) {
                        endmode = 1;
                    } else if ( msg->button.button == m_cancelb ) {
                        endmode = -1;
                    }
                    break;
                case AG_KEYPRESSED:
                    if ( msg->key.key == XK_Return ) {
                        endmode = 1;
                    } else if ( msg->key.key == XK_Escape ) {
                        endmode = -1;
                    } else if ( msg->key.key == XK_Up ) {
                        int row = m_lv->getActiveRow() - 1;
                        if ( m_lv->isValidRow( row ) ) {
                            m_lv->setActiveRow( row );
                            m_lv->showActive();
                            m_sg->setText( m_lv->getText( row, 0 ).c_str() );
                        }
                    } else if ( msg->key.key == XK_Down ) {
                        int row = m_lv->getActiveRow() + 1;
                        if ( m_lv->isValidRow( row ) ) {
                            m_lv->setActiveRow( row );
                            m_lv->showActive();
                            m_sg->setText( m_lv->getText( row, 0 ).c_str() );
                        }
                    } else if ( msg->key.key == XK_Next ) {
                        int row = m_lv->getActiveRow() + m_lv->getMaxDisplayV() - 1;

                        if ( row >= m_lv->getElements() ) row = m_lv->getElements() - 1;

                        if ( m_lv->isValidRow( row ) ) {
                            m_lv->setActiveRow( row );
                            m_lv->showActive();
                            m_sg->setText( m_lv->getText( row, 0 ).c_str() );
                        }
                    } else if ( msg->key.key == XK_Prior ) {
                        int row = m_lv->getActiveRow() - m_lv->getMaxDisplayV() + 1;

                        if ( row < 0 ) row = 0;

                        if ( m_lv->isValidRow( row ) ) {
                            m_lv->setActiveRow( row );
                            m_lv->showActive();
                            m_sg->setText( m_lv->getText( row, 0 ).c_str() );
                        }
                    } else if ( msg->key.key == XK_Tab ) {
                        switchCompletionMode();
                    }
                    break;
                case AG_FIELDLV_DOUBLECLICK:
                    if ( msg->fieldlv.lv == m_lv ) {
                        // double click in lv, actual element is unimportant here
                        endmode = 1;
                    }
                    break;
                case AG_FIELDLV_ONESELECT:
                case AG_FIELDLV_MULTISELECT:
                    if ( msg->fieldlv.lv == m_lv ) {
                        int row = m_lv->getActiveRow();
                        if ( m_lv->isValidRow( row ) ) {
                            m_sg->setText( m_lv->getText( row, 0 ).c_str() );
                        }
                    }
                    break;
                case AG_STRINGGADGET_OK:
                    if ( msg->stringgadget.sg == m_sg ) {
                        endmode = 1;
                    }
                    break;
                case AG_STRINGGADGET_CONTENTCHANGE:
                    if ( msg->stringgadget.sg == m_sg ) {
                        updateCompletionList( m_sg->getText() );

                        if ( m_sg->getCursor() > last_cursor_pos ) {
                            if ( m_lv->getElements() > 0 ) {
                                std::string s = m_lv->getText( 0, 0 );
                                m_sg->setText( s.c_str() );

                                m_sg->setSelection( m_sg->getCursor(),
                                                    s.length() );
                            }
                        }
                        last_cursor_pos = m_sg->getCursor();
                    }
                    break;
                case AG_CYCLEBUTTONCLICKED:
                    if ( msg->cyclebutton.cyclebutton == m_mode_cyb ) {
                        switchCompletionMode();
                    }
                    break;
                case AG_STRINGGADGET_DEACTIVATE:
                    m_sg->activate();
                    break;
            }
            m_aguix->ReplyMessage( msg );
        }

        if ( endmode == 1 ) {
            std::string cmd = m_sg->getText();
            bool store = true;

            if ( ! cmd.empty() ) {
                if ( cmd[0] == ' ' ) {
                    cmd.erase( 0, 1 );
                    store = false;
                }

                runCommand( cmd );

                if ( store ) {
                    if ( commandExists( cmd ) ) {
                        m_history->removeEntry( cmd );
                        m_history->pushFrontEntry( cmd );
                    } else {
                        Requester req( m_aguix );
                        req.request( "Runner error",
                                     "Command not found",
                                     "Close" );
                        endmode = 0;
                    }
                }
            }
        }
    }

    delete m_win;
    m_win = NULL;
    
    return;
}

int RunnerUI::mainLoop()
{
    if ( ! m_aguix ) return -1;

    std::string cfg = getConfigDir();

    cfg += "/program_list";
    m_programs.read( cfg );

    if ( m_show_run_dialog == false ) {
        std::list<PopUpMenu::PopUpEntry> m1;
        PopUpMenu::PopUpEntry e1;
        e1.name = "Start command:";
        e1.type = PopUpMenu::HEADER;
        m1.push_back( e1 );
        e1.type = PopUpMenu::HLINE;
        m1.push_back( e1 );

        int pos = 0;
        for ( auto it1 = m_programs.getList().begin();
              it1 != m_programs.getList().end();
              it1++ ) {
            e1.type = PopUpMenu::NORMAL;

            e1.name = it1->first;
            e1.id = pos++;
            m1.push_back( e1 );
        }

        e1.type = PopUpMenu::HLINE;
        m1.push_back( e1 );

        e1.type = PopUpMenu::NORMAL;

        e1.name = "Run...";
        e1.id = -1;
        m1.push_back( e1 );

        PopUpMenu *command_menu = new PopUpMenu( m_aguix, m1 );
        command_menu->create();

        int mouse_x, mouse_y;

        m_aguix->queryRootPointer( &mouse_x, &mouse_y );

        int root_x, root_y,
            root_width,
            root_height;

        m_aguix->getLargestDimensionOfCurrentScreen( &root_x, &root_y,
                                                     &root_width, &root_height );

        if ( mouse_x + command_menu->getWidth() >= root_x + root_width ||
             mouse_y + command_menu->getHeight() >= root_y + root_height ) {
            command_menu->show( a_max( 0, mouse_x - command_menu->getWidth() ),
                                a_max( 0, mouse_y - command_menu->getHeight() ),
                                PopUpMenu::POPUP_IN_POSITION );
        } else {
            command_menu->show();
        }

        AGMessage *msg;
        int endmode = 0;

        command_menu->setXFocus();

        for ( ; endmode == 0; ) {
            msg = m_aguix->WaitMessage( NULL );
            if ( msg != NULL ) {
                switch ( msg->type ) {
                    case AG_POPUPMENU_CLICKED:
                        if ( msg->popupmenu.menu == command_menu ) {
                            handlePopupMsg( msg );
                            endmode = -1;
                        }
                        break;
                    case AG_POPUPMENU_CLOSED:
                        if ( msg->popupmenu.menu == command_menu ) {
                            endmode = -1;
                        }
                        break;
                }
                m_aguix->ReplyMessage( msg );
            }
        }

        delete command_menu;
    } else {
        runProgramDialog();
    }

    return 0;
}

void RunnerUI::showData()
{
}

void RunnerUI::maximizeWin()
{
}

int RunnerUI::apply_filter( const std::string &filter )
{
    return 0;
}

class MyDirWalk : public NWC::DirWalkCallBack
{
public:
    MyDirWalk( std::list< std::string > &l ) : m_l( l )
    {}

    int visit( NWC::File &file, NWC::Dir::WalkControlObj &cobj )
    {
        if ( file.isReg( true ) &&
             access( file.getFullname().c_str(), X_OK ) == 0 ) {
            m_l.push_back( file.getBasename() );
        }
        return 0;
    }

    int visitEnterDir( NWC::Dir &dir, NWC::Dir::WalkControlObj &cobj )
    {
        return 0;
    }

    int visitLeaveDir( NWC::Dir &dir, NWC::Dir::WalkControlObj &cobj )
    {
        return 0;
    }

    int prepareDirWalk( const NWC::Dir &dir, NWC::Dir::WalkControlObj &cobj,
                        std::list< RefCount< NWC::FSEntry > > &return_add_entries,
                        std::list< RefCount< NWC::FSEntry > > &return_skip_entries )
    {
        return 0;
    }
private:
    std::list< std::string > &m_l;
};

std::list< std::string > RunnerUI::getCommands() const
{
    const char *path = getenv( "PATH" );

    if ( ! path ) {
        return {};
    }

    std::vector< std::string > l;

    AGUIXUtils::split_string( l, path, ':' );

    std::list< std::string > all_commands;
    std::list< std::string > unique_commands;

    MyDirWalk dircb( all_commands );

    for ( auto it1 = l.begin();
          it1 != l.end();
          it1++ ) {

        NWC::FSEntry p( *it1 );

        if ( p.isDir( true ) ) {
            NWC::Dir d( p );

            d.readDir();

            d.walk( dircb );
        }
    }

    all_commands.sort();

    std::unique_copy( all_commands.begin(),
                      all_commands.end(),
                      std::back_inserter( unique_commands ) );

    return unique_commands;
}

void RunnerUI::findCommands()
{
    if ( ! m_commands_in_path.empty() ) return;

    if ( m_win != NULL ) {
        m_win->setCursor( AGUIX::WAIT_CURSOR );
        m_aguix->Flush();
    }

    m_commands_in_path = getCommands();

    if ( m_win != NULL ) {
        m_win->unsetCursor();
    }
}

int RunnerUI::handlePopupMsg( AGMessage *msg )
{
    if ( msg == NULL ) return 1;

    if ( msg->type != AG_POPUPMENU_CLICKED ) return 1;

    std::list<int> entry_ids = *msg->popupmenu.recursive_ids;

    if ( entry_ids.empty() == true ) return 1;

    int id = entry_ids.back();

    if ( id == -1 ) {
        // run program dialog

        runProgramDialog();
    } else if ( id >= 0 ) {

        auto it1 = m_programs.getList().begin();

        for ( ; it1 != m_programs.getList().end() && id > 0;
              it1++, id-- );

        if ( it1 != m_programs.getList().end() ) {
            runCommand( it1->second );
        }
    }

    return 0;
}

void RunnerUI::switchCompletionMode()
{
    if ( m_completion_mode == HISTORY_MODE ) {
        m_completion_mode = PATH_MODE;

        m_mode_cyb->setOption( 1 );
    } else {
        m_completion_mode = HISTORY_MODE;

        m_mode_cyb->setOption( 0 );
    }

    updateCompletionList( m_sg->getText() );

    if ( m_lv->getElements() > 0 ) {
        std::string s = m_lv->getText( 0, 0 );
        m_sg->setText( s.c_str() );

        m_sg->setSelection( m_sg->getCursor(),
                            s.length() );
    }
}

void RunnerUI::setShowRunDialog( bool nv )
{
    m_show_run_dialog = nv;
}

bool RunnerUI::commandExists( const std::string &cmd )
{
    if ( m_commands_in_path.empty() ) {
        m_commands_in_path = getCommands();
    }

    std::vector< std::string > l;

    AGUIXUtils::split_string( l, cmd, ' ' );

    if ( l.empty() ) return false;

    if ( l[0].empty() ) return false;

    for ( auto &path_cmd : m_commands_in_path ) {
        if ( path_cmd == l[0] ) {
            return true;
        }
    }

    return false;
}
