#! /usr/bin/env python
# xpdfWatch.py
# Copyright (C) 2008 Ralf Hoffmann.
# ralf@boomerangsworld.de
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import re
import time
import sys
from optparse import OptionParser

XPDFWATCH_VERSION="0.4.2"

CACHE_LIFE_TIME = 10.0
CACHE_LIFE_TIME_X = 30.0

def print_version( option, opt_str, value, parser ):
    print "xpdfWatch.py %s" % ( XPDFWATCH_VERSION )
    sys.exit( 0 )

parser = OptionParser( usage = "Usage: %prog [options]" )
parser.add_option( "-d", "--debug",
                   action  = "store_true",
                   dest    = "debug",
                   default = False,
                   help    = "enable debug messages" )
parser.add_option( "-V", "--version",
                   action  = "callback",
                   callback = print_version,
                   help    = "print xpdfWatch version" )
parser.add_option( "-f", "--file",
                   action  = "store",
                   dest    = "filename",
                   type    = "string",
                   help    = "only update xpdf showing given file and quit" )
parser.add_option( "--skiptest",
                   action  = "store_true",
                   dest    = "skiptest",
                   default = False,
                   help    = "Skip test for required programs" )

( options, args ) = parser.parse_args()

modtimes = {}
xpdf_cwds = []

window_cache = {}
opened_file_cache = ( 0, [] )
opened_file_from_x_cache = ( 0, [] )
force_x_cache_update = False

def checkReqs():
    missing_progs = []

    xse_version = os.popen( "xse -version 2>&1" ).readlines()
    if len( xse_version ) < 1:
        missing_progs.append( "xse" )
    else:
        if not re.match( "^Xse 2\..*", xse_version[0] ):
            missing_progs.append( "xse" )
    
    lsof_version = os.popen( "lsof -v 2>&1" ).readlines()
    if len( lsof_version ) < 2:
        missing_progs.append( "lsof" )
    else:
        if not re.match( "\s*revision: 4\.7.*", lsof_version[1] ):
            missing_progs.append( "lsof" )

    xwininfo_version = os.popen( "xwininfo -help 2>&1" ).readlines()
    if len( xwininfo_version ) < 2:
        missing_progs.append( "xwininfo" )
    
    return missing_progs

def get_opened_xpdf_files():
    global xpdf_cwds
    global opened_file_cache
    global force_x_cache_update

    if ( time.time() - opened_file_cache[0] ) < CACHE_LIFE_TIME:
        if options.debug:
            print "DEBUG: Use list of opened files from cache"
        return opened_file_cache[1]

    lines = os.popen( "lsof -c xpdf" )
    files = []
    xpdf_cwds = []
    for l in lines:
        a = l.split()
        if a[4] == "REG" and len( a ) > 8:
            f = a[8]
            if f.endswith( ".pdf" ):
                files.append( f )
        if a[4] == "DIR" and a[3] == "cwd":
            cwd = os.path.realpath( "/proc/%s/cwd" % ( a[1].strip() ) )
            xpdf_cwds.append( cwd )

    if opened_file_cache[1] != files:
        if options.debug:
            print "DEBUG: Force update of X based file info"
        force_x_cache_update = True
    opened_file_cache = ( time.time(), files )
    return files

def get_opened_xpdf_files_from_x():
    global opened_file_from_x_cache
    global force_x_cache_update

    if ( time.time() - opened_file_from_x_cache[0] ) < CACHE_LIFE_TIME_X and force_x_cache_update == False:
        if options.debug:
            print "DEBUG: Use list of opened files (X) from cache"
        return opened_file_from_x_cache[1]

    lines = os.popen( "xwininfo -root -tree" )
    files = []
    for l in lines:
        m = re.match( '.*Xpdf: (.*)": \("win" "Xpdf"\).*', l )
        if m:
            files.append( m.group( 1 ) )
    opened_file_from_x_cache = ( time.time(), files )
    force_x_cache_update =  False
    return files

def get_changed_files():
    global modtimes
    global xpdf_cwds
    files = []

    opened_files = get_opened_xpdf_files()

    for f in get_opened_xpdf_files_from_x():
        if not os.path.isabs( f ):
            if options.debug:
                print "DEBUG: Search for", f
            for d in xpdf_cwds:
                f2 = os.path.join( d, f )
                if options.debug:
                    print "DEBUG: Candidate is", f2
                if os.path.exists( f2 ) and not f2 in opened_files:
                    if options.debug:
                        print "DEBUG: Added"
                    opened_files.append( f2 )
        else:
            if os.path.exists( f ) and not f in opened_files:
                opened_files.append( f )

    for f in opened_files:
        if os.path.exists( f ):
            try:
                m = os.path.getmtime( f )
                if f in modtimes:
                    if modtimes[f] < m:
                        size = os.path.getsize( f )
                        while True:
                            time.sleep( 1 )
                            newsize = os.path.getsize( f )
                            if newsize == size:
                                break
                            size = newsize
                        m = os.path.getmtime( f )
                        files.append( f )
                modtimes[f] = m
            except OSError:
                pass
    return files

def find_xpdf_window( file ):
    global window_cache

    search_file = os.path.basename( file )
    if search_file in window_cache and ( time.time() - window_cache[search_file][0] < CACHE_LIFE_TIME ):
        if options.debug:
            print "DEBUG: Got window for %s from cache" % ( search_file )
        return window_cache[search_file][1]

    lines = os.popen( "xwininfo -root -tree" )
    wins = []
    for l in lines:
        if re.match( '.*Xpdf: .*%s.*\("win" "Xpdf"\).*' % ( search_file ),
                     l ):
            w = l.split()[0]
            wins.append( w )
    window_cache[search_file] = ( time.time(), wins )
    return wins

def send_reload( file ):
    if options.debug:
        print "DEBUG: Finding windows for file %s" % ( os.path.basename( file ) )
    wins = find_xpdf_window( file )
    if options.debug:
        print "DEBUG: Found:", wins
    for w in wins:
        command = "xse -window %s '<KeyPress> r'" % ( w )
        if options.debug:
            print "DEBUG: Issuing %s" % ( command )
        os.system( command )

if options.skiptest == False:
    missing_progs = checkReqs()
    if len( missing_progs ) > 0:
        print "The following tools are missing or the wrong version is installed:"
        print "  ",
        for t in missing_progs:
            print t,
        print
        print "Check the xpdfWatch homepage for links to the missing programs!"
        print "Website: http://www.boomerangsworld.de/cms/tools/xpdfwatch"
        sys.exit( 5 )

if options.filename:
    send_reload( options.filename )
else:
    while True:
        files = get_changed_files()
        if options.debug:
            print "DEBUG: Changed files:", files
        for f in files:
            send_reload( f )
        time.sleep( 1 )
