/* akvp_pool.hh
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2025 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef AKVP_POOL_HH
#define AKVP_POOL_HH

#include "aguixdefs.h"
#include <string>
#include <set>
#include <memory>

namespace AKVPPool {

    class AKVPPoolInterface {
    public:
        virtual ~AKVPPoolInterface() = 0;

        struct kvp_entry {
            enum kvp_type {
                KVP_BOOLEAN,
                KVP_INT,
                KVP_STRING
            };

            kvp_entry( bool value ) :
                type( KVP_BOOLEAN ),
                bool_value( value )
            {}

            kvp_entry( int value ) :
                type( KVP_INT ),
                int_value( value )
            {}

            kvp_entry( const std::string &value ) :
                type( KVP_STRING ),
                str_value( value )
            {}

            kvp_entry( const char *value ) :
                type( KVP_STRING ),
                str_value( value )
            {}

            kvp_entry( const kvp_entry &rhs ) :
                type( rhs.type ),
                bool_value( rhs.bool_value ),
                int_value( rhs.int_value ),
                str_value( rhs.str_value )
            {}

            kvp_entry()
            {}

            enum kvp_type type = KVP_BOOLEAN;
            bool bool_value = false;
            int int_value = 0;
            std::string str_value;

            bool operator==( const kvp_entry &rhs ) const;
        };

        struct get_entry_result {
            kvp_entry value;
            kvp_entry default_value;
            bool found = false;
            bool overwritten = false;
        };
    
        virtual get_entry_result getEntry( const std::string &key ) const = 0;
        virtual void setEntry( const std::string &key,
                               kvp_entry value ) = 0;
        virtual void clear() = 0;
        virtual std::set< std::string > getKeys() const = 0;
    };

    void setGlobalKVPPool( std::shared_ptr< AKVPPoolInterface > pool );
    std::shared_ptr< AKVPPoolInterface > getGlobalKVPPool();

}

#endif /* AKVP_POOL_HH */
